/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* bundle.c
 * Provides functions for bundling kernel files
 */

#include "stddefs.h"

#if defined(ENABLE_KERNEL_BUNDLING)
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>
#include "rdev_config.h"
#include "assorted.h"
#include "str.h"

#define MAX_SANITIZED_FNAMES 256
#define STRUCTNAME "kernel_"

#define QUOTED(x) "\"" x "\""
#define CASTUCSS(x) "(unsigned char **) " x

#define STRUCTSTART(name) "struct " name " {\n"
#define STRUCTEND         "};\n"
#define STRUCTFIELD(type, name) "\t" #type " " #name ";\n"

#define TypeEnumDecl \
	"enum kerneltype_ { " \
	"KERNELTYPE_C = 0, KERNELTYPE_CUDA_C, KERNELTYPE_OPENCL_C, " \
	"KERNELTYPE_GENERIC_OUT, KERNELTYPE_CUDA_FATBIN, KERNELTYPE_CUDA_PTX, " \
	"KERNELTYPE_OPENCL_OUT, KERNELTYPE_SREC " \
	"};\n"

#define DataStructDecl STRUCTSTART(STRUCTNAME)                 \
				       STRUCTFIELD(int, id)                    \
				       STRUCTFIELD(unsigned char *, contents)  \
				       STRUCTFIELD(unsigned long, size)        \
				       STRUCTFIELD(enum kerneltype_, type)     \
				       STRUCTFIELD(char *, filename)           \
				       STRUCTFIELD(char *, primary_filename)   \
				       STRUCTEND

static str all_tableitems_inits = NULL;
static int num_files = 0;
static char *primary_file = NULL;
static char *kernel_tablename = NULL;

static
void expect_constant(char *expected, int *argc, char ***argv)
{
	char *constant;

	if ((*argc) == 0)
	{
		UNEXPECTED_VAL:
		fprintf(stderr, "ompiconf_bundle_kernel_files: error: expected %s\n", expected);
		exit(EXIT_FAILURE);
	}

	constant = (*argv)[0];

	if (strcmp(constant, expected))
		goto UNEXPECTED_VAL;

	(*argc)--;
	(*argv)++;
}


static
int expect_int(int *argc, char ***argv)
{
	int num;

	if ((*argc) == 0)
	{
		fprintf(stderr, "ompiconf_bundle_kernel_files: error: expected integer\n");
		exit(EXIT_FAILURE);
	}

	num = atoi((*argv)[0]);
	(*argc)--;
	(*argv)++;

	return num;
}


static
char *expect_str(int *argc, char ***argv)
{
	char *str;

	if ((*argc) == 0)
	{
		fprintf(stderr, "ompiconf_bundle_kernel_files: error: expected string\n");
		exit(EXIT_FAILURE);
	}

	str = (*argv)[0];
	(*argc)--;
	(*argv)++;

	return str;
}


/* Reads a file from disk and stores it to a buffer */
static
unsigned char *read_file(char *filename, size_t *filesize)
{
	FILE *fp;
	size_t size;
	unsigned char *buf;

	fp = fopen(filename, "rb");

	if (!fp)
	{
		fprintf(stderr, "read_file: error: could not open file %s\n", filename);
		fclose(fp);
		return NULL;
	}

	fseek(fp, 0L, SEEK_END);
	size = ftell(fp) + 1;

	buf = calloc((size + 1), sizeof(unsigned char));
	if (!buf)
	{
		fprintf(stderr, "read_file: error: could not allocate memory\n");
		fclose(fp);
		return NULL;
	}

	fseek(fp, 0L, SEEK_SET);
	fread(buf, sizeof(unsigned char), size, fp);
	fclose(fp);

	if (filesize)
		*filesize = size;

	return buf;
}


/* Reads a kernel file and prints its encoded format and all the
 * necessary information about it
 */
static
void print_kernel_file(int id, char *filename, char *filetype)
{
	unsigned char *buf;
	char *array, *sanitized_fname;
	size_t size;

	/* Replace all special chars of the filename with underscores */
	sanitized_fname = sanitize_str(filename);
	num_files++;
	
	/* Load the file */
	buf = read_file(filename, &size);

	/* Create the necessary C array */
	array = encode_to_hex_array(buf, filename, size);

	/* { <id>, <hexarray_name>, <size>, <type>, <filename>, <primary_filename> }; */
	str_printf(all_tableitems_inits, 
	           "\t{ %d, " 
			   "(unsigned char []) %s, "
			   "%lu, " 
			   "%s, " 
			   QUOTED("%s") ", " 
			   QUOTED("%s") " },\n", 
	           id, array, size, filetype, filename, primary_file);

	/* Free everything */
	free(buf);
	free(sanitized_fname);
	free(array);
}


static
void print_struct_table(void)
{
	int i;
	char *ext;
	char *structinits = strdup(str_string(all_tableitems_inits));

	/* Remove last comma */
	structinits[strlen(structinits) - 2] = 0;

	/* struct kerneldata_ <primary_file>[<num_files>] = {
	 *   { <struct-init> }, 
	 *   { <struct-init> }, 
	 *   ...
	 * }; 
	 */
	printf("struct " STRUCTNAME " %s[%d] = {\n%s\n};", 
	       kernel_tablename, num_files, structinits);

	free(structinits);
}


/* Bundles all kernel files into one file that contains one 
 * C declaration per file.
 */
void ompiconf_bundle_kernel_files(int *argc, char ***argv)
{
	char *filename, *filetype;
	int id, i;

	if (!all_tableitems_inits)
		all_tableitems_inits = Strnew();
	else
		str_truncate(all_tableitems_inits);

	/* First get the primary filename */
	expect_constant("--primary-file", argc, argv);
	primary_file = expect_str(argc, argv);

	/* Then get the kernel table name */
	expect_constant("--table-name", argc, argv);
	kernel_tablename = expect_str(argc, argv);

	/* Print type enum + data struct declarations */
	printf(TypeEnumDecl DataStructDecl);

	/* Expected format: --id <id> --type <type> --file <filename>... */
	while ((*argc) != 0)
	{
		expect_constant("--id", argc, argv);
		id = expect_int(argc, argv);

		expect_constant("--type", argc, argv);
		filetype = expect_str(argc, argv);

		expect_constant("--file", argc, argv);
		filename = expect_str(argc, argv);

		print_kernel_file(id, filename, filetype);
	}

	/* Print the table that contains addresses to all declared data structs */
	print_struct_table();
	
	str_free(all_tableitems_inits);
}


#endif