/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <stdio.h>
#include <stdbool.h>
#include "globals.h"
#include "locks.h"

#define MAX_LOCKS 32

/* locks.c
 * This file implements the locking mechanism
 */

__SHAREDQLFR int locks[MAX_LOCKS];
__SHAREDQLFR int used_locks;
__SHAREDQLFR int locks_initialized;

/* Should only be called by 1 thread */
__DEVQLFR 
void omp_init_lock(omp_lock_t *lock)
{
	int i;
	if (critical_lock.in_area)
	{
		if (critical_lock.mutex_init)
			return;
		critical_lock.mutex = 0;
		*lock = &(critical_lock.mutex);
		critical_lock.mutex_init = true;
		return;
	}
	if (!locks_initialized)
	{
		for (i = 0; i < MAX_LOCKS; i++)
			locks[i] = -1;
		locks_initialized = 1;
	}
	if (used_locks < MAX_LOCKS-1)
	{
		for (i = 0; i < MAX_LOCKS; i++)
		{
			if (locks[i] == -1)
			{
				*lock = &(locks[i]);
				*(int*)*lock = 0;
				used_locks++;
				return;
			}
		}
	}
	else
		*lock = NULL;
}

__DEVQLFR 
void omp_set_lock(omp_lock_t *lock)
{
	bool needlock = true;
	if (!lock) return;
	while (needlock)
	{
		if (!atomicCAS((int*) *lock, 0, 1))
		{
			needlock = false;
			return;
		}
	}
}

__DEVQLFR 
void omp_unset_lock(omp_lock_t *lock)
{
	if (!lock) return;
	atomicExch((int*) *lock, 0);
}

__DEVQLFR 
int omp_test_lock(omp_lock_t *lock)
{
	if (!lock) return 0;
	if (0 == atomicCAS((int*) *lock, 0, 1))
		return 1;
	return 0;
}

__DEVQLFR 
void omp_destroy_lock(omp_lock_t *lock)
{
	if (!lock) return;
	used_locks--;
	*(int*)*lock = -1;
}

__DEVQLFR 
void dev_lock(int *lock)
{
	while (atomicCAS(lock, 0, 1) != 0);
}

__DEVQLFR 
void dev_unlock(int *lock)
{
	if (!lock) return;
	atomicExch(lock, 0);
}
