/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef __GLOBALS_H__
#define __GLOBALS_H__

#include <device_launch_parameters.h>
#include <stdbool.h>

#if defined(CUDA)
	#define __HOSTDEVQLFR __host__ __device__
	#define __DEVQLFR __device__
	#define __CONSTQLFR __constant__
	#define __SHAREDQLFR __shared__
	#define __DEVKERN extern "C" __device__
	#define __MANAGEDQLFR __managed__
	#define __LOCALQLFR __local__
#else
	#define __HOSTDEVQLFR
	#define __DEVQLFR
	#define __CONSTQLFR
	#define __DEVKERN
	#define __SHAREDQLFR
	#define __MANAGEDQLFR
	#define __LOCALQLFR
#endif

#define MAX_ACTIVE_LEVELS 1

#define TIMES2(c) c, c
#define TIMES3(c) c, c, c

#define __NTHR          (blockDim.x * blockDim.y * blockDim.z)
#define __THRID         (threadIdx.x + (threadIdx.y * blockDim.x))
#define __BLOCKID       (blockIdx.x + blockIdx.y * gridDim.x)
#define __NBLOCKS       (gridDim.x * gridDim.y * gridDim.z)
#define __WARPID        ((__THRID) / warpSize)
#define __NWARPS        (ceilf((__NTHR) / warpSize))
#define __LASTWARP      ((__NTHR - 1) / warpSize)
#define __ISLASTWARP    ((__THRID / warpSize) >= __LASTWARP)
#define __ISWARPMASTER  (((__THRID) % warpSize) == 0)
#define __ISBLOCKMASTER ((__ISWARPMASTER) && (__ISLASTWARP))
#define __ISWORKER      (!(__ISLASTWARP))
#define __ACTIVEWARPS   (ceilf(dev_get_parallel_active_threads() / warpSize))

/* CUDA initialization-related macros */
#define INIT_MASTERWORKER 0
#define INIT_COMBINED     1

/* CUDA control block macros */
#define TS_SIZE 1024
#define __MYCB        ((__THRID < TS_SIZE) ? &(thread_cbs[__THRID]) : NULL)

#define SHMEM_STACK_MAXSIZE 128
#define SHMEM_STACK_INITIALIZED 1
typedef struct shmem_
{
	void *stack[SHMEM_STACK_MAXSIZE];
	int size;
	int init;
} shmem_t;

typedef struct 
{
	int max_teams;
	int max_threads;
	int nthr;
	int thread_limit;
	int nprocs;
} cuda_thrinfo_t;

/* CUDA initialization-related macros */
#define INIT_MASTERWORKER 0
#define INIT_COMBINED     1

/* CUDA control block macros */
#define TS_SIZE 1024
typedef struct thread_cb_
{
	bool intask;
	bool in_nested_parallel;
} thread_cb_t;

typedef struct shared_data_
{
    int inmasterworker;
} shared_data_t;

extern __CONSTQLFR cuda_thrinfo_t thrinfo;
extern __SHAREDQLFR shared_data_t shdata;
extern __SHAREDQLFR thread_cb_t thread_cbs[TS_SIZE];

extern __DEVQLFR int  omp_get_thread_num(void);
extern __DEVQLFR int  omp_get_num_threads(void);
extern __DEVQLFR int  omp_get_max_active_levels(void);
extern __DEVQLFR void omp_set_num_threads(int num_threads);
extern __DEVQLFR int  omp_is_initial_device(void);
extern __DEVQLFR int  omp_get_num_teams(void);
extern __DEVQLFR int  omp_get_team_num(void);
extern __DEVQLFR int  omp_get_max_threads(void);
extern __DEVQLFR int  omp_get_max_teams(void);
extern __DEVQLFR int  omp_get_thread_limit(void);
extern __DEVQLFR int  omp_get_level(void);
extern __DEVQLFR int  omp_get_active_level(void);
extern __DEVQLFR int  omp_get_num_procs(void);
extern __DEVQLFR int  omp_get_dynamic(void);
extern __DEVQLFR int  omp_get_nested(void);
extern __DEVQLFR int  omp_in_parallel(void);

extern __SHAREDQLFR shmem_t cuda_shmem;

#endif
