/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* OMPICC
 * A driver for the OMPi compiler.
 */

/*
 * May 2019:
 *   Code related to modules/devices and kernels moved to kernels.c.
 * July 2016:
 *   Small refactoring and more documentation.
 * Oct 2010:
 *   Improvements; correct argument quoting.
 * Apr 2010:
 *   Some improvements
 * Aug 2007:
 *   Major simplifications due to new threadprivate implementation.
 *   Removed POMP support.
 * May 2007:
 *   Fixed some definitions for compiling/linking/preprocessing.
 *   -lrt now determined at configuration time.
 *   Some less important bug fixes.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#if defined(HAVE_REALPATH)
	#include <limits.h>
#endif

#include "config.h"
#include "stddefs.h"
#include "git_version.h"
#include "str.h"
#include "keyval.h"
#include "ompicc.h"
#include "kernels.h"

#ifdef OMPI_REMOTE_OFFLOADING
	#include "rdev_config.h"
#endif


/* This function uses system(3) to execute the shell command pointed by 'cmd'.
 * On success, the termination status of the child process which is retrieved
 * using WIFEXITED and WEXITSTATUS as described in waitpid(2) is returned.
 * If the child process couldn't be created or its status couldn't be retrieved
 * a) if 'exit_flag' is NULL -1 is returned; 
 * b) if 'exit_flag' is not NULL, exit_flag is set to -1. If system(3) is 
 *    successful and 'exit_flag' is not NULL, exit_flag is set to 0.
 */
int sysexec(char *cmd, int *exit_flag)
{
	int ret;

	/* If the child process could not be created or its status
	 * could not be retrieved set exit_flag to 1.
	 */
	if ((ret = system(cmd)) == -1)
	{
		if (exit_flag)
			return (*exit_flag = -1);
		_exit(-1);
	}
	if (exit_flag) 
		*exit_flag = 0;
	return (WIFEXITED(ret) ? WEXITSTATUS(ret) : ret);
}


/* How does ompicc decide on the flags to use when processing user programs?
 *
 * First, ompicc knows the flags used to build OMPi. These are hard-coded
 * at the time ompicc is built and include all the flags required to build
 * the last threading library. These are provided in Makefile.am and include
 * CompileFlags, LinkFlags, IncludeDir, LibDir etc. See comment below for
 * a full list.
 *
 * Second, ompicc reads the OMPI_XXX environmental variables. If any of
 * them exists, it overrides the corresponding hard-coded flags above.
 * In the end, the values of PREPROCESSOR, COMPILER, CPPFLAGS, CFLAGS,
 * LDFLAGS and ORTINFO variables below are either set to the hard-coded flags
 * or the corresponding environmental variable. These are the ones used to
 * build the user program.
 *
 * Finally, the user may pass additional flags when running ompicc. These
 * are stored in the variables named "user_xxx" (a bit below) and are
 * used as *additional* flags.
 *
 * Esoteric note:
 *   Because the hard-coded flags are the ones used to build
 *   the last threading library, there may be linker flags needed if the user
 *   utilizes another library with --ort.
 *   Should fix some day :-(
 */

/* Definitions (macros) provided extrernally (see Makefile.am).
 * These are hard-coded when making ompicc and are the defaults.
 * They can be user-overriden through the OMPI_XXX environmental variables.
 * The last two are overriden if a portable build of OMPi is in effect.
 *
 * OmpiName
 * OmpiConfName
 * CPPcmd
 * CCcmd
 * PreprocFlags
 * CompileFlags
 * LinkFlags
 * IncludeDir
 * LibDir
 */

/* Flags collected from the OMPI_CPP, OMPI_CPPFLAGS,
 * OMPI_CC, OMPI_CFLAGS and OMPI_LDFLAGS, from the ./configure info and the
 * library-specific configuration file.
 */
char PREPROCESSOR[PATHSIZE], COMPILER[PATHSIZE],
     CPPFLAGS[FLAGSIZE], CFLAGS[FLAGSIZE], LDFLAGS[FLAGSIZE],
     ORTINFO[FLAGSIZE];
#ifdef OMPI_REMOTE_OFFLOADING
	char COMMINFO[FLAGSIZE];
#endif
char ortlibname[PATHSIZE],           /* The runtime library needed */
     commlibname[PATHSIZE],
     RealOmpiName[PATHSIZE];

static char **availortlibs;
static int    nortlibs;

#ifdef PORTABLE_BUILD
	char InstallPath[PATHSIZE], LibDir[PATHSIZE], IncludeDir[PATHSIZE];
#endif

#ifdef OMPI_REMOTE_OFFLOADING
	#define ompi_info() \
		fprintf(stderr, \
		"%s %s using:\n  >> system compiler: %s\n  >> runtime library: %s\n"\
		                    "  >> communication library: %s\n"\
							"  >> config. devices: %s\n",\
		PACKAGE_NAME, GIT_VERSION, COMPILER, *ORTINFO ? ORTINFO : ortlibname,\
			*COMMINFO ? COMMINFO : ortlibname, MODULES_CONFIG)
#else
	#define ompi_info() \
		fprintf(stderr, \
		"%s %s using:\n  >> system compiler: %s\n  >> runtime library: %s\n"\
							"  >> config. devices: %s\n",\
		PACKAGE_NAME, GIT_VERSION, COMPILER, *ORTINFO ? ORTINFO : ortlibname,\
			MODULES_CONFIG)
#endif

#define ompi_shortinfo() \
	fprintf(stderr, "%s %s \n", PACKAGE_NAME, GIT_VERSION)


static void ompi_help(char *ompibin)
{
	fprintf(stderr, "\nUsage: %s [ompi options] [system compiler options] programfile(s)\n",
			ompibin);
	fprintf(stderr, "\n"
			"   Useful OMPi options:\n"
			"              --help: display this information\n"
			"           --options: show all OMPi options\n"
			"       --info [what]: OpenMP-related help\n"
			"                  -k: keep intermediate file\n"
			"                  -v: be verbose (show the actual steps)\n"
			"        --ort=<name>: use a specific OMPi runtime library\n"
#ifdef OMPI_REMOTE_OFFLOADING
			"       --comm=<name>: use a specific remote offloading comm library\n"
#endif
			"    --devs=<devices>: target the given devices\n"
			"  --devopt [options]: set device options\n"
			"\n"
			"Use environmental variables OMPI_CPP, OMPI_CC, OMPI_CPPFLAGS,\n"
			"OMPI_CFLAGS, OMPI_LDFLAGS to have OMPi use a particular base\n"
			"preprocessor and compiler, along with specific flags.\n");
}

static void ompicc_error(int exitcode, char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	fprintf(stderr, "[ompicc error]: ");
	vfprintf(stderr, format, ap);
	va_end(ap);

	exit(exitcode);
}


static char *get_basename(char *path)
{
	char *s;

	if (path == NULL || *path == 0)
		return ".";
	else
		if (path[0] == '/' && path[1] == 0)
			return path;
		else
		{
			s = path;
			while (*s)
				s++;
			s--;
			while (*s == '/' && s != path)   /* destructive */
				*s-- = 0;
			if (s == path)
				return path;
			while (*s != '/' && s != path)
				s--;
			if (*s == '/')
				return s + 1;
			else return path;
		}
}


static arg_t *new_arg(char opt, char *val)
{
	arg_t *p;

	if ((p = (arg_t *) malloc(sizeof(arg_t))) == NULL)
		ompicc_error(-1, "malloc() failed\n");
	p->opt = opt;
	if (val != NULL)
		strcpy(p->val, val);
	else p->val[0] = 0;
	p->next = NULL;
	return p;
}


void arglist_add(arglist_t *l, arg_t *arg)
{
	if (l->head == NULL)
		l->head = l->tail = arg;
	else
	{
		l->tail->next = arg;
		l->tail = arg;
	}
}


int append_arg(arglist_t *l, int argc, char **argv, int proceed)
{
	char opt, val[SLEN];
	arg_t *p;

	val[0] = 0;
	if (argv[0][0] == 0)
		return 0;
	if (argv[0][0] != '-')
	{
		p = new_arg(0, *argv);
		arglist_add(l, p);
		return 0;
	}
	opt = argv[0][1];

	if (argv[0][2] != 0)
	{
		strcpy(val, &argv[0][2]);
		p = new_arg(opt, val);
		arglist_add(l, p);
		return 0;
	}
	else
	{
		if (proceed && argc > 1)
			strcpy(val, &argv[1][0]);
		p = new_arg(opt, val);
		arglist_add(l, p);
		return proceed && argc > 1;
	}
}


static int quotedlen(char *s)
{
	int len;
	for (len = 0; *s != 0; s++)
		len += (*s == '"' ? 2 : 1);
	return (len);
}


/* Prepare a string from an argument list; all args are quoted */
static void strarglist(char *dest, arglist_t *l, int maxlen)
{
	arg_t *p;
	char  *c, *d = dest;

	for (*d = 0, p = l->head; p != NULL; p = p->next)
	{
		if ((d - dest) + quotedlen(p->val) + 6 >= maxlen)
			ompicc_error(1, "argument(s) too long; rebuild OMPi with larger LEN.\n");
		if (p->opt)
		{
			snprintf(dest, maxlen, "-%c ", p->opt);
			dest += ((p->opt == 'o') ? 3 : 2);
		}
		*(dest++) = '"';
		for (c = p->val; *c != 0; c++)
		{
			if (*c == '"') *(dest++) = '\\';
			*(dest++) = *c;
		}
		*(dest++) = '"';
		*(dest++) = ' ';
	}
	*dest = 0;
}


int fok(char *fname)
{
	struct stat buf;

	return (stat(fname, &buf) == 0);
}

/* Debuggers */
#define DBG_NONE     0
#define DBG_GDB      1
#define DBG_VALGRIND 2

bool disableOpenMP = false,
     disableOmpix = false,
     cppLineNo = false;
bool mustlink = true;     /* Becomes 0 if -c parameter is specified */
bool makefile = true;     /* Becomes 0 if --nomakefile parameter is specified */
int  keep = 0;            /* Becomes 1/2 if -k/K parameter is specified */
bool verbose = false;
bool showdbginfo = false; /* Force the compiler to show some debugging info */
bool showdevinfo = false; /* Display info about configured devices */
bool longdevinfo = false; /* short by default */
int  usegdb = DBG_NONE;   /* If > 0, _ompi will be run through a debugger */
bool usevalgrind = false; /* If true _ompi will be run through gdb. */
bool usecarstats = false;
bool reductionOld = false; /* Make _ompi produce newer style reduction code */
bool reductionRTL = false; /* New reduction code style selector */
bool autoscope = false;    /* Enable autoscoping dfa */
int  taskoptlevel = 1;     /* default; never used by ompicc, though */
char *reqmodules = NULL;   /* Requested modules by the user */
int  reqjobs = -1;

#if defined(ENABLE_KERNEL_BUNDLING)
static char *kernel_tablename = NULL;
static char *kbundlefname = NULL;
bool onlykernelfile = false; /* Compile only a single kernel file */
int  _kernid = 0;
#endif

arglist_t user_files     = { NULL, NULL };  /* Files to be compiled/linked */
arglist_t user_outfile   = { NULL, NULL };  /* Output, -o XXX */
arglist_t user_prep_args = { NULL, NULL };  /* Preprocessor args */
arglist_t user_link_args = { NULL, NULL };  /* Linker args given by the user */
arglist_t user_scc_flags = { NULL, NULL };  /* Remaining system compiler args */


/*
 * Options
 */


#define OMPI_OPTNAME(opt) "-" #opt
#define OPTNAME(opt)      "--" #opt
#define OPTNAME_V(opt)    "V--" #opt "="
#define OPTION(opt)       OPT_##opt

typedef enum {
	OPTION(unknown) = -1, /* unknown option */
	
	OPTION(dummy1),  /* Info/help options */
	OPTION(version),     OPTION(help),        OPTION(options),     
	OPTION(info),
	
	OPTION(dummy2),  /* Code generation and runtime options */
	OPTION(ort),         OPTION(devs),        OPTION(devopt),
	OPTION(taskopt),     OPTION(reduction),   OPTION(comm),

	OPTION(dummy4),  /* Debug/inspect options */
	OPTION(verbose),     OPTION(keep),        OPTION(keepall),
	OPTION(nolineno),    OPTION(dbg),
	
	OPTION(dummy5),  /* Feature options */
	OPTION(scopecheck),  OPTION(carstats),
	
	OPTION(dummy6),  /* Developer options */
	OPTION(nomp),        OPTION(nox),         OPTION(nomakefile),
	OPTION(gdb),         OPTION(valgrind),    OPTION(complete),
	OPTION(kernel),
	
	OPTION(lastoption)    /* marker */
} option_t;

char *optnames[] = {
	NULL,
	OPTNAME(version),     OPTNAME(help),      OPTNAME(options),     
	OPTNAME(info),

	NULL,
	OPTNAME_V(ort),       OPTNAME_V(devs),    OPTNAME(devopt),
	OPTNAME_V(taskopt),   OPTNAME_V(reduction), OPTNAME_V(comm),

	NULL,
	OPTNAME(verbose),     OPTNAME(keep),        OPTNAME(keepall),
	OPTNAME(nolineno),    OPTNAME(dbg),
	
	NULL,
	OPTNAME(scopecheck),  OPTNAME(carstats),
	
	NULL,
	OPTNAME(nomp),        OPTNAME(nox),         OPTNAME(nomakefile),
	OPTNAME(gdb),         OPTNAME(valgrind),    OPTNAME_V(complete),
	OPTNAME_V(kernel),
	
	NULL
};


/*
 * _ompi options
 */
 
typedef struct _ompi_opt_
{
	option_t id;
	char    *name;
	bool     used;
} _ompi_opt_t;

str _ompi_optstr; /* Contains all _ompi arguments */

/* ompicc <-> _ompi option mapping */
_ompi_opt_t _ompi_opts[] = {
	{ OPTION(taskopt),    OMPI_OPTNAME(taskopt),     false },
	{ OPTION(reduction),  "",                        false },
	{ OPTION(nolineno),   OMPI_OPTNAME(nolineno),    false },
	{ OPTION(dbg),        OMPI_OPTNAME(showdbginfo), false },
	{ OPTION(scopecheck), OMPI_OPTNAME(autoscope),   false },
	{ OPTION(carstats),   OMPI_OPTNAME(drivecar),    false },
	{ OPTION(nomp),       OMPI_OPTNAME(nomp),        false },
	{ OPTION(nox),        OMPI_OPTNAME(nox),         false },
	{ OPTION(lastoption), NULL,                      false }
};

/* Marks an _ompi option as `used' */
static void _ompi_opt_setused(option_t oid)
{
	int i;
	for (i = 0; _ompi_opts[i].id != OPTION(lastoption); i++)
		if (_ompi_opts[i].id == oid)
		{
			_ompi_opts[i].used = true;
			return;
		}
}

/* Builds the _ompi option string, which is passed 
 * when running _ompi */
static void _ompi_build_optstr(char *fname)
{
	int i;
	
	/* (1) Reset */
	str_truncate(_ompi_optstr);
	
	/* (2) Check OMPi CFLAGS and append _ompi options, accordingly */
	if (strstr(CFLAGS, "OMPI_MAIN=LIB"))
		str_printf(_ompi_optstr, " -nomain ");
    
	if (strstr(CFLAGS, "OMPI_MEMMODEL=PROC+THR"))
		str_printf(_ompi_optstr, " -procs -threads ");   
	else 
		if (strstr(CFLAGS, "OMPI_MEMMODEL=PROC"))
			str_printf(_ompi_optstr, " -procs ");
		else 
			if (strstr(CFLAGS, "OMPI_MEMMODEL=THR"))
				str_printf(_ompi_optstr, " -threads ");
	
	/* (3) Iterate through all used ompicc options and append
	 * the corresponding _ompi ones */
	for (i = 0; _ompi_opts[i].id != OPTION(lastoption); i++)
	{
		if (_ompi_opts[i].used && _ompi_opts[i].name) 
		{
			switch (_ompi_opts[i].id)
			{
				/* Append the taskopt level */
				case OPTION(taskopt):
					str_printf(_ompi_optstr, " %s%d ", _ompi_opts[i].name, taskoptlevel);
					break;
				/* Requires manual handling */
				case OPTION(reduction):
					if (reductionOld)
						str_printf(_ompi_optstr, " -oldred ");
					else if (reductionRTL)
						str_printf(_ompi_optstr, " -rtlibred ");
					break;
				/* Every other argument */
				default:
					str_printf(_ompi_optstr, " %s ", _ompi_opts[i].name);
					break;
			}
		}
	}
	
	/* (4) Always include the `--usemod' options */
	str_printf(_ompi_optstr, " %s ", modules_argfor_ompi());

#ifdef ENABLE_KERNEL_BUNDLING
	kernel_tablename = bundle_get_tablename(fname);
	str_printf(_ompi_optstr, " --kerneltable=%s ", kernel_tablename);
#endif
}


char *taskargnames[] = { "size", "speed", "speed+", NULL };
char *redargnames[]  = { "old", "default", "rtlib", NULL };
char *infoargnames[]  = { "env", "directive", "clause", "find", NULL };

char *optinfo[] = {
	"Info/help options",
	"", 
	"         (show short OMPi options info)",
	"      (show all OMPi options)", 
	" [args]  (show OpenMP-related help)",

	"Code generation and runtime options",
	"<eelib>                   (threading library to use)",
	"<module,module,...>      (target only the named device modules)",
	" [options]              (set device options)",
	"[size|speed|speed+]   (default: speed)", 
	"[default|rtlib|old]",
	"<commlib>                (communication library to use for remote offloading)",

	"Debug/inspect options",
	" (detail every step of the compilation process)",
	"    (keep main intermediate file)",
	" (keep all intermediate files)",
	"     (disable # <line> in intermediate files)",
	"          (force compiler to display some debug info)", 
	
	"Feature options",
	"   (enable autoscoping analysis)", 
	"     (turn on C.A.R.S. kernel analysis)",

	"Developer options",
	"         (disable OpenMP)", 
	"          (disable OMPi extensions)",
	"", 
	"          (runs _ompi from within gdb)", 
	"     (runs _ompi from within valgrind)", 
	NULL,
	NULL,

	NULL
};

option_t optid(char *arg, char **val)
{
	int i;

	for (i = 0; i < OPTION(lastoption); i++)
	{
		if (!optnames[i])   /* Skip dummy options */
			continue;
		if (optnames[i][0] == 'V')     /* Option with value */
		{
			if (strncmp(optnames[i]+1, arg, strlen(optnames[i])-1) == 0)
			{
				*val = arg + strlen(optnames[i]) - 1;
				return ((option_t) i);
			}
		}
		else
			if (strcmp(optnames[i], arg) == 0)
				return ((option_t) i);
	}
	return ( OPTION(unknown) );
}


void showallopts()
{
	int i;
	char *realopt;
	ompi_shortinfo();
	fprintf(stderr, "\n[ AVAILABLE OPTIONS ]\n");
	for (i = 0; i < OPTION(lastoption); i++)
		if (optnames[i])
		{
			realopt = optnames[i]+(optnames[i][0]=='V' ? 1 : 0);
			if (strcmp(realopt, OPTNAME(verbose)) == 0)
				fprintf(stderr, "  -v | ");
			else if (strcmp(realopt, OPTNAME(keepall)) == 0)
				fprintf(stderr, "  -K | ");
			else if (strcmp(realopt, OPTNAME(keep)) == 0)
				fprintf(stderr, "  -k | ");
			else if (strcmp(realopt, OPTNAME_V(complete)+1) == 0)
				continue;
			else if (strcmp(realopt, OPTNAME_V(kernel)+1) == 0)
				continue;
			else
				fprintf(stderr, "  ");
			fprintf(stderr, "%s%s\n", realopt, optinfo[i]);
		}
		else
			fprintf(stderr, "\n%s\n%.*s\n", optinfo[i], (int) strlen(optinfo[i]), 
			       "------------------------------------------------------");
}


void helpOpenMPenv(char *what)
{
	int i, max = 0;
	static char *ompenv[][2] = {
		{ "[ OpenMP ]", "" },
		{ "  OMP_DYNAMIC",           "boolean" },
		{ "  OMP_NESTED",            "boolean" },
		{ "  OMP_SCHEDULE",          "policy[,int]" },
		{ "",                        "     policy=STATIC|DYNAMIC|GUIDED|AUTO" },
		{ "  OMP_STACKSIZE",         "int[C]     C=B|K|M|G (default:K)" },
		{ "  OMP_THREAD_LIMIT",      "int" },
		{ "  OMP_MAX_ACTIVE_LEVELS", "int" },
		{ "  OMP_WAIT_POLICY",       "ACTIVE|PASSIVE" },
		{ "  OMP_NUM_THREADS",       "int[,int[,int ...]]" },
		{ "  OMP_PROC_BIND",         "TRUE|FALSE|<list of types>" },
		{ "",                        "     types=MASTER|CLOSE|SPREAD" },
		{ "  OMP_CANCELLATION",      "boolean" },
		{ "  OMP_DISPLAY_ENV",       "TRUE|FALSE|VERBOSE" },
		{ "  OMP_PLACES",            "symbolic[(int)]|<list of places>" },
		{ "",                        "     symbolic=THREAD|CORES|SOCKETS" },
		{ "  OMP_DEFAULT_DEVICE",    "int" },
		{ "  OMP_MAX_TASK_PRIORITY", "int" },
		{ "  OMP_DISPLAY_AFFINITY",  "boolean" },
		{ "  OMP_AFFINITY_FORMAT",   "string" },
		{ "  OMP_TARGET_OFFLOAD",    "MANDATORY|DISABLED|DEFAULT" },
		{ "\n[ OMPi global ]", "" },
		{ "  OMPI_DYNAMIC_TASKQUEUESIZE", "boolean     (default:FALSE)" },
		{ "  OMPI_TASK_DEQUEUE_POLICY",   "FIFO|LIFO   (default:FIFO)" },
		{ "  OMPI_PAR2TASK_POLICY",  "TRUE|FALSE|AUTO  (default:AUTO)" },
		{ "  OMPI_HOSTTARGET_SHARE", "boolean     (default:FALSE)" },
		{ "  OMPI_SCHED_AUTO",       "string      (default:\"static\")" },
		{ "  OMPI_TAG_SCHED_<TAG>",  "string      (default:\"static\")" },
		{ "\n[ OMPi pthreads library ]", "" },
		{ "  OMPI_PTHREADS_POLICY",  "KILL|POOL|PREPOOL (default:POOL)" },
		{ "  OMPI_PTHREADS_MAXTHR",  "int    (max #persistent threads)" },
		{ "\n[ OMPi devices ]", "" },
		{ "  OMPI_CUDA_ARCH_TARGET", "string      (default:\"\")" },
		{ NULL, NULL }
	};
	
	if (what != NULL)   /* search for a particular string */
	{
		for (i = max = 0; ompenv[i][0] != NULL; i++)
			if (strstr(ompenv[i][0], what) != NULL || strstr(ompenv[i][1], what))
			  max = 1;
		if (max)
		{
			fprintf(stderr, "--- OpenMP environmental variables ---\n");
			for (i = 0; ompenv[i][0] != NULL; i++)
				if (strstr(ompenv[i][0], what) != NULL || strstr(ompenv[i][1], what))
					fprintf(stderr, "%s :  %s\n", ompenv[i][0], ompenv[i][1]);
		}
		return;
	}
	
	ompi_shortinfo();
	fprintf(stderr, "\nAll runtime env. variables\n--------------------------\n");
	for (i = 0; ompenv[i][0] != NULL; i++)
		if (strlen(ompenv[i][0]) > max)
			max = strlen(ompenv[i][0]);
	for (i = 0; ompenv[i][0] != NULL; i++)
		fprintf(stderr, "%-*s  %s\n", max, ompenv[i][0], ompenv[i][1]);
}


void helpOpenMPshow()
{
	fprintf(stderr, "\n[ Arguments for OpenMP-related help ]\n");
	fprintf(stderr, "  --info env              "
	       " (shows all OpenMP and OMPi environmental variables)\n");
	fprintf(stderr, "  --info directive <name> "
	       " (shows all clauses allowed in the given directive)\n");
	fprintf(stderr, "  --info clause <name>    "
	       " (shows all directives accepting the given clause)\n");
	fprintf(stderr, "  --info find <string>    "
	       " (shows OpenMP entities containing the given string)\n");
}


void helpOpenMP(int argc, char **argv)
{
	str cmd = Strnew();
	
	ompi_shortinfo();
	if (argc == 1)
	{
		HELPERROR:
		helpOpenMPshow();
		return;
	}
	argc--; argv++;
	if (strcmp(*argv, "env") == 0)
	{
		helpOpenMPenv(NULL);
		return;
	}
	if (argc == 1)
		goto HELPERROR;
	
	if (strncmp(*argv, "dir", 3) == 0)
		str_printf(cmd, "%s __internal__ dirclauses %s", RealOmpiName, argv[1]);
	else if (strncmp(*argv, "clau", 4) == 0)
		str_printf(cmd, "%s __internal__ clausedirs %s", RealOmpiName, argv[1]);
	else if (strncmp(*argv, "find", 4) == 0)
		str_printf(cmd, "%s __internal__ searchomp %s", RealOmpiName, argv[1]);
	else
		goto HELPERROR;
	
	fprintf(stderr, "[ OpenMP help/information about %s ]\n\n", argv[1]);
	if (strncmp(*argv, "find", 4) == 0)
		helpOpenMPenv(argv[1]);
	sysexec(str_string(cmd), NULL);
	str_free(cmd);
}

/* Device options */
#define DEVOPTNAME(opt)   "--" #opt
#define DEVOPTNAME_V(opt) "V--" #opt "="
#define DEVOPTION(opt)    DEVOPT_##opt

static bool devoptsused = false;

typedef enum {
	DEVOPTION(unknown) = -1, /* unknown option */
	DEVOPTION(dummy1),  /* Info/help options */
	DEVOPTION(jobs),
	DEVOPTION(lastoption)    /* marker */
} devoption_t;

char *devoptnames[] = {
	NULL,
	DEVOPTNAME_V(jobs),
	NULL
};

static
void showdevopts()
{
	fprintf(stderr, "\nSyntax: --devopt [options]\n");
	fprintf(stderr, "\n[ Available device options ]\n");
	fprintf(stderr, "  -j[N] | --jobs[=N] "
		     "(allow N jobs for kernel makefiles; infinite jobs with no argument.)\n");
}

static
int clamp(int x, int min, int max)
{
	return ((x > max) ? (max) : ((x < min) ? (min) : (x)));
}

static
bool is_number(char *str)
{
	int i;
	for (i = 0; i < strlen(str); i++)
		if (!isdigit(str[i]))
			return false;
	return true;
}

option_t devoptid(char *arg, char **val, bool *valexists)
{
	int i;
	*valexists = false;

	for (i = 0; i < DEVOPTION(lastoption); i++)
	{
		if (!devoptnames[i])   /* Skip dummy options */
			continue;
		if (devoptnames[i][0] == 'V')     /* Option with value */
		{
			if (strncmp(devoptnames[i]+1, arg, strlen(devoptnames[i])-1) == 0)
			{
				*val = arg + strlen(devoptnames[i]) - 1;
				*valexists = true;
				return ((devoption_t) i);
			}
			else if (strncmp(devoptnames[i]+1, arg, strlen(devoptnames[i])-2) == 0)
				return ((devoption_t) i);
		}
		else
			if (strcmp(devoptnames[i], arg) == 0)
				return ((devoption_t) i);
	}
	return ( DEVOPTION(unknown) );
}

void handle_devopt(int argc, char **argv)
{
	char *val, *optval;
	bool valexists = false;

	if (devoptsused) 
		ompicc_error(1, "currently the `--devopt' flag cannot be used multiple times.\n");

	if (argc == 1)
		goto UNSUPPORTED_DEVOPT;

	/* Skip "--devopt" */
	argv++;
	val = strdup(argv[0]);

	switch ( devoptid(val, &optval, &valexists) )
	{
		case DEVOPTION(jobs):
		{
			if (!valexists) 
			{
				reqjobs = 0;
				break;
			}

			if (strlen(optval) == 0 || !is_number(optval)) 
				goto UNSUPPORTED_DEVOPT;

			reqjobs = clamp(atoi(optval), 0, 99);
			break;
		}		
		default:
		{
			if (val[0] == '-')
			{
				switch (val[1])
				{
					case 'j':
					{
						if (strlen(val) == 2)
						{
							reqjobs = 0;
							break;
						}

						if (!is_number(val = val+2)) 
							goto UNSUPPORTED_DEVOPT;

						reqjobs = clamp(atoi(val), 0, 99);
						break;
					}
					default:
						goto UNSUPPORTED_DEVOPT;
						break;
				}
			}
			else 
			{
				UNSUPPORTED_DEVOPT:
				ompi_shortinfo();
				showdevopts();
				_exit(1);
			}	
			break;
		}
	}	

	devoptsused = true;
}


void parse_args(int argc, char **argv)
{
	int  d, ortlib = 0, commlib = 0, oid, dev_oid;
	char *parameter, *val, *compval, *ompibin = argv[0];
	void get_ort_flags();
	void get_comm_flags();

	argv++;
	argc--;

	while (argc)
	{
		d = 0;
		switch ( oid = optid(parameter = argv[0], &val) )
		{
			case OPTION(complete):
				modules_employ(NULL);
				int i;
				switch ( dev_oid = optid(val, &compval) )
				{
					/* Autocomplete --devs=... */
					case OPTION(devs):
					{
						for (i = 0; i < nmodules; i++)
							if (strncmp(modulenames[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, modulenames[i]);
						break;						
					}
					/* Autocomplete --taskopt=... */
					case OPTION(taskopt):
					{
						for (i = 0; taskargnames[i] != NULL; i++)
							if (strncmp(taskargnames[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, taskargnames[i]);
						break;
					}
					/* Autocomplete --reduction=... */
					case OPTION(reduction):
					{
						for (i = 0; redargnames[i] != NULL; i++)
							if (strncmp(redargnames[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, redargnames[i]);
						break; 
					}
					/* Autocomplete --ort=... */
					case OPTION(ort):
					{
						for (i = 0; i < nortlibs; i++)
							if (strncmp(availortlibs[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, availortlibs[i]);
						break;
					}
					default:
					{
						bool valempty = false;
						char *letteropts = "clLIDUokKv";
						
						for (i = 0; i < strlen(val); i++)
						{
							if (isspace(val[i])) 
							{
								valempty = true;
								break;
							}
						}
						if (val[0] == '-' && strlen(val) <= 2)
						{
							if (strlen(val) == 1)
							{
								for (i = 0; i < strlen(letteropts); i++)
									fprintf(stderr, "-%c\n", letteropts[i]);
							}
							else if (strchr(letteropts, val[1]) && strcmp(val, "--"))
							{
								fprintf(stderr, "%s\n", val);
								break;
							}
						}
						for (i = 0; i != OPTION(lastoption); i++)
						{
							if (!optnames[i])
								continue;

							if (!strncmp(optnames[i]+(optnames[i][0] == 'V' ? 1 : 0), val, strlen(val)) 
							|| valempty)
								fprintf(stderr, "%s\n", 
									optnames[i][0] == 'V' ? optnames[i]+1 : optnames[i]);
						}
						
						break;
					}
				}
				_exit(0);
				break;
			case OPTION(help):
				strcpy(ortlibname, "default");
				strcpy(commlibname, "default_comm");
				get_ort_flags();
#ifdef OMPI_REMOTE_OFFLOADING
				get_comm_flags();
#endif
				ompi_info();
				ompi_help(ompibin);
				_exit(0);
				break;
			case OPTION(version):
				fprintf(stderr, "%s\n", GIT_VERSION);
				_exit(0);
				break;
			case OPTION(options):
				showallopts();
				_exit(0);
				break;
			case OPTION(devopt):
				handle_devopt(argc, argv);
				argc--; argv++;
				break;
			case OPTION(info):
				helpOpenMP(argc, argv);
				_exit(0);
				break;
			case OPTION(ort):
				strncpy(ortlibname, val, 511);
				ortlib = 1;
				break;
			case OPTION(comm):
				strncpy(commlibname, val, 511);
				commlib = 1;
				break;
#if defined(ENABLE_KERNEL_BUNDLING)
			case OPTION(kernel):     
				onlykernelfile = true;
				_kernid = atoi(val);
				break;
#endif
			case OPTION(keep):
				goto KEEP_MAIN;
			case OPTION(keepall):
				goto KEEP_ALL;
			case OPTION(verbose):
				goto BEVERB;
			case OPTION(nomp):       disableOpenMP = true; break;
			case OPTION(nox):        disableOmpix = true; break;
			case OPTION(nomakefile): makefile = false; break;
			case OPTION(nolineno):   cppLineNo = true; break;
			case OPTION(gdb):        usegdb = DBG_GDB; break;
			case OPTION(valgrind):   usegdb = DBG_VALGRIND; break;
			case OPTION(dbg):        showdbginfo = true; break;
			case OPTION(carstats):   usecarstats = true; break;
			case OPTION(scopecheck): autoscope = true; break;
			case OPTION(taskopt):
				if (strcmp(val, "size") == 0)
					taskoptlevel = 0;
				else
					if (strcmp(val, "speed") == 0)
						taskoptlevel = 1;
					else
						if (strcmp(val, "speed+") == 0)
							taskoptlevel = 2;
						else
							ompicc_error(1,"unknown taskopt option (try 'size'/'speed'/'speed+').\n");
				break;
			case OPTION(devs):
				reqmodules = val;
				break;
			case OPTION(reduction):
				if (strcmp(val, "old") == 0)
					reductionOld = true;
				else
					if (strcmp(val, "rtlib") == 0)
					{
						reductionOld = false;
						reductionRTL = true;
					}
					else 
						if (strcmp(val, "default") == 0)
						{
							reductionOld = false;
							reductionRTL = false;
						}
						else
						  ompicc_error(1,"unknown reduction option (try 'old'/'rtlib').\n");
						  
				break;
			default:
				if (parameter[0] == '-')              /* option */
					switch (parameter[1])
					{
						case 'c':
							mustlink = false;
							break;
						case 'l':
							d = append_arg(&user_link_args, argc, argv, 1);
							break;
						case 'L':
							d = append_arg(&user_link_args, argc, argv, 1);
							break;
						case 'I':
							d = append_arg(&user_prep_args, argc, argv, 1);
							break;
						case 'D':
							d = append_arg(&user_prep_args, argc, argv, 1);
							break;
						case 'U':
							d = append_arg(&user_prep_args, argc, argv, 1);
							break;
						case 'o':
							d = append_arg(&user_outfile, argc, argv, 1);
							break;
						case 'k':
							KEEP_MAIN:
							keep = 1; d = 0;
							break;
						case 'K':
							KEEP_ALL:
							keep = 2; d = 0;
							break;
						case 'v':
							BEVERB:
							verbose = true; d = 0;
							break;
						default:
							d = append_arg(&user_scc_flags, argc, argv, 0);
							break;
					}
				else
				{
					d = append_arg(&user_files, argc, argv, 0);
					if (!fok(user_files.tail->val))
						ompicc_error(1, "file %s does not exist\n", user_files.tail->val);
				};
		}
		
		/* Mark option as "used" */
		if ((oid > OPTION(unknown)) && (oid < OPTION(lastoption)))
			_ompi_opt_setused(oid);
			
		argc = argc - 1 - d;
		argv = argv + 1 + d;
	}
	
	if (!ortlib)
		strcpy(ortlibname, "default");    /* Default lib to use */
	if (!commlib)
		strcpy(commlibname, "default_comm");    /* Default lib to use */
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        COMPILING                                          *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


char *compile_prefix()
{
	static char prefix[LEN+64];
	
	if (usegdb != DBG_NONE)
	{
		if (usegdb == DBG_GDB)
			sprintf(prefix, "gdb %s -ex 'set arg", RealOmpiName);
		else
			sprintf(prefix, "valgrind %s", RealOmpiName);
		return prefix;
	}
	else
		return RealOmpiName;
}

char *compile_suffix()
{
	if (usegdb != DBG_NONE)
	{
		if (usegdb == DBG_GDB)
			return "'";
		else
			return "";
	}
	else
		return "";
}


void ompicc_compile(char *fname)
{
	char *s, *compfmt, preflags[SLEN], noext[PATHSIZE], outfile[PATHSIZE];
	char strscc_flags[LEN], strgoutfile[LEN];
	str cmd = Strnew();
	int  res, eflag;
	int  nkernels = 0;  /* # kernels in user code */

	if ((s = strrchr(fname, '.')) != NULL)
		if (strcmp(s, ".o") == 0) return;     /* skip .o files */

	strcpy(noext, fname);
	if ((s = strrchr(noext, '.')) != NULL) *s = 0; /* remove ext */
	snprintf(outfile, PATHSIZE, "%s_ompi.c", noext);

	/* Preprocess
	 */
	strarglist(preflags, &user_prep_args, SLEN);
#if defined(__SYSOS_cygwin) && defined(__SYSCOMPILER_cygwin)
	/* Hack for CYGWIN gcc */
	str_printf(cmd, "%s -U__CYGWIN__ -D__extension__=  -U__GNUC__ "
#ifdef OMPI_OMP_EXT
                    " -D_OMPI_OMP_EXT "
#endif
#ifdef OMPI_REMOTE_OFFLOADING
                     " -D_OMPI_REMOTE_OFFLOADING "
#endif
	                   " %s -I%s %s \"%s\" > \"%s.pc\"",
	        PREPROCESSOR, CPPFLAGS, IncludeDir, preflags, fname, noext);
#else
	/* We need to explicitly #define _OMPI_OMP_EXT because it is checked inside
	 * <omp.h> (and possibly other user-facing files, too)
	 */
	str_printf(cmd, "%s -U__GNUC__ %s "
#ifdef OMPI_OMP_EXT
                     " -D_OMPI_OMP_EXT "
#endif
#ifdef OMPI_REMOTE_OFFLOADING
                     "-D_OMPI_REMOTE_OFFLOADING "
#endif
	                "-I%s %s \"%s\" > \"%s.pc\"",
	        PREPROCESSOR, CPPFLAGS, IncludeDir, preflags, fname, noext);
#endif
	if (verbose)
		fprintf(stderr, "====> Preprocessing file (%s.c)\n  [ %s ]\n", noext, str_string(cmd));
	if ((res = sysexec(str_string(cmd), NULL)) != 0)
		_exit(res);

	/* Transform
	 */

	_ompi_build_optstr(fname);
		
#ifdef PORTABLE_BUILD
	compfmt = "%s \"%s.pc\" __ompi__ \"%s\"%s > \"%s\"%s";
#else
	compfmt = "%s \"%s.pc\" __ompi__%s > \"%s\"%s";
#endif

	str_truncate(cmd);
	str_printf(cmd, compfmt,
	        compile_prefix(RealOmpiName),
	        noext,
#ifdef PORTABLE_BUILD
	        InstallPath,             /* Use as the first argument */
#endif
	        str_string(_ompi_optstr),
	        outfile,
	        compile_suffix());
	if (verbose)
		fprintf(stderr, "====> Transforming file (%s.c)\n  [ %s ]\n", noext, str_string(cmd));
	res = sysexec(str_string(cmd), NULL);
	if (keep < 2)
	{
		str_truncate(cmd);
		str_printf(cmd, "%s.pc", noext);               /* remove preprocessed file */
		unlink(str_string(cmd));
	}
	if (res == 33)                                /* no pragma omp directives */
	{
		FILE *of = fopen(outfile, "w");
		if (of == NULL)
		{
			fprintf(stderr, "Cannot write to intermediate file.\n");
			_exit(1);
		}
		if (cppLineNo)
			fprintf(of, "# 1 \"%s.c\"\n", noext); /* Identify the original file */
		fclose(of);
		str_truncate(cmd);
		str_printf(cmd, "cat \"%s.c\" >> \"%s\"", noext, outfile);
		if (sysexec(str_string(cmd), &eflag))
		{
			unlink(outfile);
			_exit(res);
		}
	}
	else
	{
		if (res != 0)
		{
			if (!keep)
				unlink(outfile);
			_exit(res);
		}
		else   /* All OK - parse the 2nd line to get needed info */
		{
			FILE *of = fopen(outfile, "r");
			char ch = 0;
			int  nk;

			if (of != NULL)
			{
				for (ch = fgetc(of); ch!=EOF && ch!='\n'; ch = fgetc(of))
					;
				if (ch == '\n' && fscanf(of, "$OMPi__nfo:%d", &nk) == 1)
					nkernels = nk;
				fclose(of);
			}
		}
	}

#if defined(ENABLE_KERNEL_BUNDLING) && (KERNEL_BUNDLING_MODE == BUNDLE_SOURCES)
	if (nkernels && (nmodules || rdev_config.nuniquemodules))
	{
		if (verbose)
			fprintf(stderr, "====> Bundling kernel files for %d module(s)\n",
			                nmodules + rdev_config.nuniquemodules);
		kbundlefname = bundle_kernel_files(fname, kernel_tablename, nkernels);
	}
#endif

	/* Compile the transformed file
	 */
	strarglist(strscc_flags, &user_scc_flags, LEN);
	str_truncate(cmd);
	str_printf(cmd, "%s \"%s\" -c %s -I%s %s %s",
	        COMPILER, outfile, CFLAGS, IncludeDir, preflags, strscc_flags);

	if (verbose)
		fprintf(stderr, "====> Compiling file (%s):\n  [ %s ]\n", outfile, str_string(cmd));
	res = sysexec(str_string(cmd), &eflag);
	if (!keep)
		unlink(outfile);
	if (eflag || res)
		_exit(res);

	/* Settle the output file name
	 */
	strarglist(strgoutfile, &user_outfile, LEN);
	strcpy(noext, get_basename(fname));
	if ((s = strrchr(noext, '.')) != NULL) * s = 0; /* remove ext */
	if (user_outfile.head != NULL && !mustlink)
		strcpy(outfile, user_outfile.head->val);
	else
		snprintf(outfile, PATHSIZE, "%s.o", noext);
	strcat(noext, "_ompi.o");
	if (verbose)
		fprintf(stderr, "====> Renaming file \"%s\" to \"%s\"\n",
		        noext, outfile);
	rename(noext, outfile);

#if defined(OMPI_REMOTE_OFFLOADING) && defined(USE_STATIC_MPI_PROCS)
	if (verbose)
		fprintf(stderr, "====> Generating %s-mpirun.sh script\n", PACKAGE_TARNAME);
	str_truncate(cmd);
	str_printf(cmd, "%sconf __internal__ --rdev-action=\"create-mpirun-script\"",
	                PACKAGE_TARNAME);
	res = sysexec(str_string(cmd), &eflag);
	if (res)
	{
		fprintf(stderr, "could not create mpirun script\n");
		_exit(res);
	}
#endif

	/* Kernels must be created within the directory of the source file.
	 * If all source files are in the same directory we are doing
	 * redundant copies here.
	 */
#if defined(OMPI_REMOTE_OFFLOADING)
	if (nkernels && (nmodules || rdev_config.nuniquemodules))
	{
		if (verbose)
			fprintf(stderr, "====> Generating kernel makefiles for %d module(s)\n",
			                nmodules + rdev_config.nuniquemodules);
		kernel_makefiles(fname, nkernels); /* 99.9% get_basename didn't touch it */
	}
#else
	if (nkernels && nmodules)
	{
		if (verbose)
			fprintf(stderr, "====> Generating kernel makefiles for %d module(s)\n",
			                nmodules);
		kernel_makefiles(fname, nkernels); /* 99.9% get_basename didn't touch it */
	}
#endif

#if defined(ENABLE_KERNEL_BUNDLING) && (KERNEL_BUNDLING_MODE == BUNDLE_BINARIES)
	if (nkernels && (nmodules || rdev_config.nuniquemodules))
	{
		if (verbose)
			fprintf(stderr, "====> Bundling kernel files for %d module(s)\n",
			                nmodules + rdev_config.nuniquemodules  );
		kbundlefname = bundle_kernel_files(fname, kernel_tablename, nkernels);
	}
#endif

	str_free(cmd);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        LINKING                                            *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


void ompicc_link()
{
	arg_t *p;
	char cur_obj[PATHSIZE], tmp[PATHSIZE];
	str cmd = Strnew();
	char objects[LEN], *obj;
	char strsccargs[LEN], strlinkargs[LEN], strgoutfile[LEN], strprepargs[LEN];
	int len, is_tmp, eflag;
	char rm_obj[LEN];

	obj = objects;
	*obj = 0;
	strcpy(rm_obj, "rm -f ");
	for (p = user_files.head; p != NULL; p = p->next)
	{
		strcpy(cur_obj, p->val);
		is_tmp = 0;
		len = strlen(cur_obj);
		if (cur_obj[len - 1] == 'c')
			is_tmp = 1;
		if (is_tmp)
		{
			cur_obj[len - 2] = 0;
			strcpy(tmp, cur_obj);
			strcpy(cur_obj, get_basename(tmp));
			strcat(cur_obj, ".o");
			strcat(rm_obj, cur_obj);
			strcat(rm_obj, " ");
		}
		snprintf(obj, LEN, "\"%s\" ", cur_obj);
		obj += strlen(cur_obj) + 3;
	}

	strarglist(strsccargs,  &user_scc_flags, LEN);
	strarglist(strlinkargs, &user_link_args, LEN);
	strarglist(strgoutfile, &user_outfile, LEN);
	strarglist(strprepargs, &user_prep_args, LEN);

	/* We have to include -lort 2 times due to circular dependencies
	 * with the threading libraries.
	 */
#ifdef OMPI_REMOTE_OFFLOADING
	str_printf(cmd,
	        "%s %s %s -I%s %s %s %s -L%s -L%s/%s -L%s/%s -lort -lcomm %s %s -lort -lcomm",
	        COMPILER, objects, CFLAGS, IncludeDir, strprepargs, strsccargs,
	        strgoutfile, LibDir, LibDir, ortlibname, LibDir, commlibname, LDFLAGS, strlinkargs);
#else
	str_printf(cmd,
	        "%s %s %s -I%s %s %s %s -L%s -L%s/%s -lort %s %s -lort",
	        COMPILER, objects, CFLAGS, IncludeDir, strprepargs, strsccargs,
	        strgoutfile, LibDir, LibDir, ortlibname, LDFLAGS, strlinkargs);
#endif

#if defined(ENABLE_KERNEL_BUNDLING)
	if (kbundlefname)
		str_printf(cmd, " \"%s\"", kbundlefname);
#endif
	if (verbose)
		fprintf(stderr, "====> Linking:\n  [ %s ]\n", str_string(cmd));
	if (sysexec(str_string(cmd), &eflag))
		fprintf(stderr, "Error: could not perform linking.\n");
	sysexec(rm_obj, &eflag);   /* Remove unnecessary files */
	str_free(cmd);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *                  Generate Makefile                        *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


void replace_variable(char *line, char *variable, char *replace_with)
{
	char *p, tmp[SLEN];

	while ((p = strstr(line, variable)) != NULL)
	{
		snprintf(tmp, SLEN, "%s%s", replace_with, p + strlen(variable));
		//Should check if p - line + strlen(tmp) > sizeof(line)
		strcpy(p, tmp);
	}
}


/* Prepare a string from an argument list; all args are quoted;
 * copied from strarglilst()
 */
static void getargs(char *dest, arglist_t *l, int maxlen)
{
	arg_t *p;
	char  *c, *d = dest;

	for (*d = 0, p = l->head; p != NULL; p = p->next)
	{
		if ((d - dest) + quotedlen(p->val) + 6 >= maxlen)
			ompicc_error(1, "argument(s) too long; rebuild OMPi with larger LEN.\n");
		for (c = p->val; *c != 0; c++)
		{
			if (*c == '"') *(dest++) = '\\';
			*(dest++) = *c;
		}
		if (p->next)
			*(dest++) = ' ';
	}
	*dest = 0;
}


void ompicc_makefile(char *compilerPath)
{
	char strfiles[LEN], strgoutfile[LEN];
	char line[SLEN];
	str infile = Strnew();
	FILE *fp, *infp;

	/* Check if the library has a template Makefile
	 */
	str_printf(infile, "%s/%s/MakefileTemplate", LibDir, ortlibname);
	if (!(infp = fopen(str_string(infile), "r")))
	{
		makefile = false;
		str_free(infile);
		return;
	}
	str_free(infile);

	getargs(strfiles, &user_files, LEN);

	//The file ompi will output
	if (user_outfile.head && strlen(user_outfile.head->val) != 0)
		snprintf(strgoutfile, LEN, "%s", user_outfile.head->val);
	else
		snprintf(strgoutfile, LEN, "a.out");

	if (verbose)
		fprintf(stderr, "====> Outputing Makefile\n");
	if ((fp = fopen("Makefile", "w")) == NULL)
		fprintf(stderr, "Error: could not generate Makefile\n");
	else
	{
		fprintf(fp, "# Makefile generated by %s\n", PACKAGE_NAME);

		while (fgets(line, sizeof line, infp) != NULL)
		{
			replace_variable(line, "@OMPICC@", compilerPath);
			replace_variable(line, "@OMPI_INPUT@", strfiles);
			replace_variable(line, "@OMPI_OUTPUT@", strgoutfile);
			replace_variable(line, "@OMPI_ORTLIB@", ortlibname);
			fputs(line, fp);  /* write the line */
		}

		fclose(fp);
	}
}


void ompicc_makefile_compile()
{
	char cmd[LEN];
	int res;

	/* Run the compile target */
	snprintf(cmd, LEN, "make compile");  /* make should be on the user PATH */

	if (verbose)
		fprintf(stderr, "====> Running target compile on generated Makefile\n");

	if ((res = sysexec(cmd, NULL)) != 0)
		_exit(res);
}


void ompicc_makefile_link()
{
	char cmd[LEN];
	int res;

	/* Run the link target */
	snprintf(cmd, LEN, "make link");  /* make should be on the user PATH */

	if (verbose)
		fprintf(stderr, "====> Running target link on generated Makefile\n");

	if ((res = sysexec(cmd, NULL)) != 0)
		_exit(res);
}

#if defined(ENABLE_KERNEL_BUNDLING)
void ompicc_compile_kernel(char *fname)
{
	kernel_makefile_single(fname);
}
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        THE MAIN() PART                                    *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Get values from environmental variables if they exist,
 * otherwise grab the default configuration values (as hard-coded
 * from Makefile.am)
 */
void ompicc_get_envvars()
{
	char *t;

	if ((t = getenv("OMPI_CPP")) == NULL)
		strncpy(PREPROCESSOR, CPPcmd, 511);
	else
		strncpy(PREPROCESSOR, t, 511);

	if ((t = getenv("OMPI_CPPFLAGS")) == NULL)
		strncpy(CPPFLAGS, PreprocFlags, 511);
	else
		strncpy(CPPFLAGS, t, 511);

	if ((t = getenv("OMPI_CC")) == NULL)
		strncpy(COMPILER, CCcmd, 511);
	else
		strncpy(COMPILER, t, 511);

	if ((t = getenv("OMPI_CFLAGS")) == NULL)
		strncpy(CFLAGS, CompileFlags, 511);
	else
		strncpy(CFLAGS, t, 511);

	if ((t = getenv("OMPI_LDFLAGS")) == NULL)
		strncpy(LDFLAGS, LinkFlags, 511);
	else
		strncpy(LDFLAGS, t, 511);
}

static
void get_available_ortlibs()
{
	int i;
	char *ortstr, *s;
	ortstr = strdup("default " ORTLIBS_CONFIG);
	
	for (; isspace(*ortstr) || *ortstr == ','; ortstr++)
		;
	if (*ortstr == 0)
		return;
	for (nortlibs = 1, s = ortstr; *s; s++)
	{
		if (isspace(*s) || *s == ',')
		{
			for (*s = 0, s++; isspace(*s) || *s == ','; s++)
				*s = ',';  /* all spaces become commas */
			if (*s)
				nortlibs++;
			s--;
		}
	}

	if ((availortlibs = (char **) malloc(nortlibs*sizeof(char *))) == NULL)
	{
		fprintf(stderr, "cannot allocate memory");
		exit (1);
	}

	for (i = 0, s = ortstr; i < nortlibs; i++)
	{
		for (availortlibs[i] = s++; *s; s++)
			;
		if (i == nortlibs-1)
			break;
		for (; *s == 0 || *s == ','; s++)
			;
	}
}

/* Read the runtime library's configuration file
 */
void get_ort_flags()
{
	str confpath = Strnew();
	FILE *fp;
	void setflag(char *, char *, void *);

	str_printf(confpath, "%s/%s/ortconf.%s", LibDir, ortlibname, ortlibname);
	if ((fp = fopen(str_string(confpath), "r")) == NULL)
		ompicc_error(1, "library `%s' cannot be found\n  (%s is missing)\n",
		             ortlibname, str_string(confpath));
	keyval_read(fp, setflag, NULL);
	fclose(fp);
	str_free(confpath);
}


#ifdef OMPI_REMOTE_OFFLOADING
/* Read the communication library's configuration file
 */
void get_comm_flags()
{
	str confpath = Strnew();
	FILE *fp;
	void setflag(char *, char *, void *);

	str_printf(confpath, "%s/%s/commconf.%s", LibDir, commlibname, commlibname);
	if ((fp = fopen(str_string(confpath), "r")) == NULL)
		ompicc_error(1, "library `%s' cannot be found\n  (%s is missing)\n",
		             commlibname, str_string(confpath));
	keyval_read(fp, setflag, NULL);
	fclose(fp);
	str_free(confpath);
}
#endif


void setflag(char *key, char *value, void *ignore)
{
	if (strcmp(key, "ORTINFO") == 0 && strlen(value) + strlen(ORTINFO) < FLAGSIZE)
		strcat(*ORTINFO ? strcat(ORTINFO, " ") : ORTINFO, value);
#ifdef OMPI_REMOTE_OFFLOADING
	if (strcmp(key, "COMMINFO") == 0 && strlen(value) + strlen(COMMINFO) < FLAGSIZE)
		strcat(*COMMINFO ? strcat(COMMINFO, " ") : COMMINFO, value);
#endif
	if (strcmp(key, "CPPFLAGS") == 0 && strlen(value) + strlen(CPPFLAGS) < FLAGSIZE)
		strcat(strcat(CPPFLAGS, " "), value);
	if (strcmp(key, "CFLAGS") == 0 && strlen(value) + strlen(CFLAGS) < FLAGSIZE)
		strcat(strcat(CFLAGS, " "), value);
	if (strcmp(key, "LDFLAGS") == 0 && strlen(value) + strlen(LDFLAGS) < FLAGSIZE)
		strcat(strcat(LDFLAGS, " "), value);
	if (strcmp(key, "CC") == 0 && strlen(COMPILER) < FLAGSIZE)
		strcpy(COMPILER, value);
	if (strcmp(key, "CPP") == 0 && strlen(PREPROCESSOR) < FLAGSIZE)
		strcpy(PREPROCESSOR, value);
}


void get_path(char *argv0, char *path)
{
	int i;

	memset(path, '\0', PATHSIZE);

	for (i = strlen(argv0); i >= 0; i--)
	{
		if (argv0[i] == '/')
		{
			strncpy(path, argv0, i + 1);
			path[i + 1] = '\0';
			break;
		}
	}

}


int main(int argc, char **argv)
{
	arg_t *p;
#if defined(HAVE_REALPATH)
	char  argv0[PATHSIZE];
	char  path[PATHSIZE];
	char  *res;
	
	_ompi_optstr = Str("");
	strcpy(argv0, "");
	res = realpath(argv[0], argv0);
	if (res == NULL)
		strcpy(RealOmpiName, OmpiName);
	else
	{
		get_path(argv0, path);         /* path before ompicc */
		strcpy(RealOmpiName, path);
		strcat(RealOmpiName, OmpiName);
	}
#else
	strcpy(RealOmpiName, OmpiName);
#endif

#ifdef PORTABLE_BUILD
	/* Code for making ompi portable in Linux. This wont work on other OSes.
	 */
	char buffer[PATHSIZE];
	ssize_t len = readlink("/proc/self/exe", buffer, PATHSIZE - 1);
	if (len == -1)
		ompicc_error(1, "couldn't retrieve installation path using readlink.\n");
	else
		if (len == PATHSIZE - 1)
			ompicc_error(1, "path to %s too long.\n", PACKAGE_TARNAME);
		else
			buffer[len] = '\0';

	/* Get the path and also get rid of the trailing "bin/" */
	get_path(buffer, InstallPath);
	if (strcmp(InstallPath + strlen(InstallPath) - 4, "bin/"))
		ompicc_error(1, "invalid installation path for a portable build.\n");
	InstallPath[strlen(InstallPath) - 4] = 0;
	
	if (strlen(InstallPath) + 8 + strlen(PACKAGE_TARNAME) + 1 > PATHSIZE)
		ompicc_error(1, "path to %s too long.\n", PACKAGE_TARNAME);

	strcpy(LibDir, InstallPath);
	strcat(LibDir, "lib/");
	strcat(LibDir, PACKAGE_TARNAME);
	strcpy(IncludeDir, InstallPath);
	strcat(IncludeDir, "include/");
	strcat(IncludeDir, PACKAGE_TARNAME);
#endif
	
	get_available_ortlibs();
	ompicc_get_envvars();
	parse_args(argc, argv);
	get_ort_flags();
#ifdef OMPI_REMOTE_OFFLOADING
	get_comm_flags();
#endif
	modules_employ(reqmodules);
	
#ifdef OMPI_REMOTE_OFFLOADING
	remote_modules_employ();
#endif
	
	if (argc == 1)
	{
		ompi_info();
		ompi_help(argv[0]);
		exit(0);
	}

	if (verbose)
	{
		ompi_info();
		fprintf(stderr, "----\n");
	}

	if (user_files.head == NULL)
	{
		fprintf(stderr, "No input file specified; "
		                "run %s with no arguments for help.\n", argv[0]);
		exit(0);
	}

#if defined(ENABLE_KERNEL_BUNDLING)
	if (onlykernelfile)
	{
		ompicc_compile_kernel(user_files.head->val);
		exit(0);
	}
#endif

	if (makefile)
		ompicc_makefile(argv[0]);

	/* If makefile exists
	 */
	if (makefile)
	{
		ompicc_makefile_compile();

		if (mustlink)
			ompicc_makefile_link();
	}
	else
	{
		for (p = user_files.head; p != NULL; p = p->next)
			ompicc_compile(p->val);
		if (mustlink)
			ompicc_link();
	}
	
#ifdef OMPI_REMOTE_OFFLOADING
	remote_modules_finalize();
#endif

	str_free(_ompi_optstr);

	return (0);
}
