/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_task_cu.c -- transform CUDA task constructs */

#include "ast_xform.h"
#include "ast_copy.h"
#include "ast_free.h"
#include "ast_renlabs.h"
#include "x_task.h"
#include "x_task_cu.h"
#include "x_clauses.h"
#include "x_target.h"
#include "ompi.h"
#include "outline.h"
#include "ast_assorted.h"

static astexpr xt_callsite_xtraargs_cuda;
astexpr xt_callsite_expr_cuda(symbol func, astexpr funcargs)
{
	/* Add the extra parameters */
	funcargs = funcargs ? CommaList(funcargs, xt_callsite_xtraargs_cuda) : 
	                      xt_callsite_xtraargs_cuda;
	return FunctionCall(
		       IdentName("_ort_new_task"),
		       CommaList(Identifier(func), funcargs)
		     );
}

static
void _omp_task_cuda(aststmt *t, taskopt_e opt)
{
	static int cuda_tasknum = 0;

	aststmt   body_copy, parent, v;
	aststmt   deparray = NULL;                          /* For dependencies */
	astexpr   ifexpr = NULL, finalexpr = NULL, prioexpr = NULL;
	ompclause c;
	bool      untied, mergeable;
	int       nindeps, noutdeps, ninoutdeps;
	outcome_t op;
	char      clabel[22];

	static outline_opts_t oo =
	{
		/* structbased             */  true,              
		/* functionName            */  "",                
		/* functionCall  (func)    */  xt_callsite_expr_cuda,  
		/* byvalue_type            */  BYVAL_bycopy,      
		/* byref_type              */  BYREF_pointer,     
		/* byref_copyptr (2 funcs) */  NULL, NULL,        
		/* global_byref_in_struct  */  false,             
		/* structName              */  "__taskenv__",     
		/* structVariable          */  "_tenv",           
		/* structInitializer       */  NULL,              
		/* implicitDefault (func)  */  xt_implicitDefault,
		/* deviceexpr              */  NULL,              
		/* addComment              */  false,             
		/* thestmt                 */  NULL,
		/* userType                */  NULL                 
	};

	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	
	/* The name of the label used for canceling. We use line number to avoid
	 * conflicts
	 */
	snprintf(clabel, 22, "CANCEL_task_%d", (*t)->u.omp->l);

	/* We insert the label before creating a copy, to prevent doing it twice and
	 * to make sure that the label gets renamed correctly (in case mergeable is
	 * present)
	 */
	if ((*t)->u.omp->body->type == COMPOUND)
		ast_stmt_append((*t)->u.omp->body->body,
		                 Labeled(Symbol(clabel), Expression(NULL)));
	else
		(*t)->u.omp->body = Compound(
		                      BlockList(
		                        (*t)->u.omp->body,
		                        Labeled(Symbol(clabel), Expression(NULL))
		                      )
		                    );

	body_copy = ast_stmt_copy((*t)->u.omp->body);         /* Keep a full copy */

	/* (1) Take care of special clauses:
	 *     - Check for untied, mergeable, if and final clauses and keep a copy
	 *     - Prepare a dependence array, if corresponding clauses are present
	 */
	if ((c = xc_ompcon_get_unique_clause((*t)->u.omp, OCIF)) != NULL)
		ifexpr = ast_expr_copy(c->u.expr);
	if ((c = xc_ompcon_get_unique_clause((*t)->u.omp, OCFINAL)) != NULL)
		finalexpr = ast_expr_copy(c->u.expr);
	if ((c = xc_ompcon_get_unique_clause((*t)->u.omp, OCPRIORITY)) != NULL)
		prioexpr = ast_expr_copy(c->u.expr);
	untied = (xc_ompcon_get_clause((*t)->u.omp, OCUNTIED) != NULL);
	mergeable = (xc_ompcon_get_clause((*t)->u.omp, OCMERGEABLE) != NULL);
	deparray = xt_assemble_deps((*t)->u.omp, &nindeps, &noutdeps, &ninoutdeps);

	/* (2) Outline
	 */
	sprintf(oo.functionName, "_cudataskFunc%d_", cuda_tasknum++);
	/* (struct structType *) _ort_taskenv_alloc(sizeof(struct), functionName); */
	oo.structInitializer =
	  CastedExpr(
	    Casttypename(
	      SUdecl(SPEC_struct, Symbol(oo.structType), NULL, NULL),
	      AbstractDeclarator(Pointer(), NULL)
	    ),
	    FunctionCall(
	      IdentName("_ort_taskenv_alloc"),
	      CommaList(
	        Sizeoftype(Casttypename(
	                     SUdecl(SPEC_struct, Symbol(oo.structType), NULL, NULL),
	                     NULL
	                   )),
	        IdentName(oo.functionName)
	      )
	    )
	  );
	xt_callsite_xtraargs_cuda = 
		Comma8(
			ifexpr ? ast_expr_copy(ifexpr) : ZeroExpr(),           /* if expression */
			finalexpr ? ast_expr_copy(finalexpr) : ZeroExpr(),    /* finalxpression */
			numConstant(untied ? 1 : 0),                       /* tied(0)/untied(1) */
			prioexpr ? ast_expr_copy(prioexpr) : ZeroExpr(),       /* priority expr */
			deparray ? Identifier(Symbol(DEPARRAY)) 
			         : CastedExpr(
			             Casttypename(
			               Declspec(SPEC_void), 
			               AbstractDeclarator(
			                 Speclist_right(
			                   Declspec(SPEC_star), Declspec(SPEC_star)), NULL)
			             ), 
			             ZeroExpr()
			           ),/* dependence array */
			numConstant(noutdeps),                                      /* #outdeps */
			numConstant(nindeps),                                        /* #indeps */
			numConstant(ninoutdeps)                                   /* #inoutdeps */
		);
	oo.thestmt = *t;
	
	op = outline_OpenMP(t, oo);
	
	if (deparray)        /* Add dependence array after the struct declaration */
	{
		if (op.repl_struct)
			ast_stmt_append(op.repl_struct, deparray);
		else 
			/* Ideal:  ast_compound_insert_statement(op.replacement, deparray);
			 * Fact:   op.replacement is not always a compound.
			 * Result:
			 */
			ast_stmt_prepend(op.repl_funcall, deparray);
	}
	
	parent = op.replacement->parent;

	/* (3) Generate the code 
	 */
	switch (opt)
	{
		case OPT_NONE: 
			*t = gencode_noopt(ifexpr, finalexpr, mergeable, deparray != NULL,
			                   body_copy, &op);
			break;
		case OPT_FAST:
		case OPT_ULTRAFAST:
			*t = gencode_fast(ifexpr, finalexpr, mergeable, deparray != NULL,
			                   body_copy, &op);
			break;
	}
	
	*t = BlockList(v, *t);         /* Add comment */
	ast_stmt_parent(parent, *t);

	//If we used a struct for byref/byvalue variables call _ort_taskenv_free
	if (op.func_struct)
		ast_stmt_prepend(op.func_return,
		                  FuncCallStmt(
		                    IdentName("_ort_taskenv_free"),
		                    CommaList(
		                      IdentName(oo.structName),
		                      IdentName(oo.functionName)
		                    )
		                  )
		                 );
}
void xform_task_cuda(aststmt *t)
{
	if (targetTask)
	{
		xform_task(t);
		return;
	}
	ifmaster_stmt(&((*t)->u.omp->body), false);
	
	xform_ompcon_body((*t)->u.omp);
	_omp_task_cuda(t, taskoptLevel);	
}