/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* assorted.h */

#ifndef __ASSORTED_H__
#define __ASSORTED_H__

#include <stdbool.h>
#include <sys/types.h>

/* 
 * JOBS -- simple functions for doing work in parallel (using processes)
 * 
 * Available macros:
 * 
 *   JOB_START(jobname, nprocs, worksize): 
 *     Starts a job of `worksize' iterations, named `jobname', that will be executed by 
 *     `nprocs' processes.
 * 
 *   JOB_LOOP(jobname, itername): 
 *     Forms a loop for `jobname', using an iteration variable named `itername',
 *     with each process receiving a chunk of a total of `worksize' iterations.
 * 
 *   JOB_FINISH(jobname): 
 *     Kills a job named `jobname'.
 */

enum execstatus_ { JOB_STOPPED = 0, JOB_RUNNING, JOB_KILLED };

typedef struct jobcb_
{
	int myid;
	pid_t pid;
	int nprocs;
	int worksize;
	enum execstatus_ status;
	unsigned long fi, li;
} jobcb_t;

#define JOB_START(jobname, nprocs, worksize) \
	jobcb_t * jobname = job_new(nprocs, worksize); \
	if (!jobname)                                  \
	{                                              \
		fprintf(stderr, "JOB_START: error: invalid worksize\n");\
		exit(EXIT_FAILURE);                        \
	}                                              \
	job_run(jobname);

#define JOB_LOOP(jobname, itername) \
	unsigned long itername = 0; \
	if (job_get_chunk(jobname)) \
		for (itername = jobname->fi; itername < jobname->li; itername++)

#define JOB_FINISH(jobname) job_kill(jobname);

extern int job_get_chunk(jobcb_t *job);
extern jobcb_t *job_new(int nprocs, int worksize);
extern void job_run(jobcb_t *job);
extern void job_kill(jobcb_t *job);

/* 
 * ENCODING/DECODING 
 */

enum encstaticflag_ {
	NOT_STATIC = 0, STATIC
};

enum encprintflag_ {
	DONT_PRINT_SIZE = 0, PRINT_SIZE
};

extern char *create_string_decl(char *buffer, char *array_name, unsigned long nbytes);
extern char *encode_to_hex_array_withdecl(unsigned char *buffer, char *array_name, unsigned long nbytes,
                                 bool static_decl, bool print_size);
extern char *encode_to_hex_array(unsigned char *buffer, char *array_name, unsigned long nbytes);

extern int is_substr(char *str, char *substr);
extern char *sanitize_str(char *str);
extern bool contains_word(const char *str, const char *word) ;

#endif /* __ASSORTED_H__ */