/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ompiconf.c
 * Provides information about the OMPi installation (currently device-only info)
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "config.h"
#include "modules.h"
#include "stddefs.h"
#include "assorted.h"
#include "bundle.h"


#define ompi_info() \
	fprintf(stderr, \
	 "OMPi-conf, the %s module helper:\n  >>  config. devices: %s\n",\
	  PACKAGE_NAME, MODULES_CONFIG)

static void ompi_help(char *binary_name)
{
	fprintf(stderr, "\nUsage: %s [ompiconf options]\n",
			binary_name);
	fprintf(stderr, "\n"
			"   Useful OMPi-conf options:\n"
			"             --help: display this information\n"
			"          --devinfo: show short info about configured devices\n"
			"         --devvinfo: show verbose log about configured devices\n"
			"     -v / --verbose: enable verbose output (show the actual steps)\n"
			"\n");
}

bool checkmodule = false;
bool showdevinfo = false;
bool verbose = false;
bool internal = false;
char *module_query = NULL;
char *rdev_query = NULL;

/* Remote device query flags */
bool getnprocs = false;
bool getmpihosts = false;
bool creatempirunscript = false;

#define OMPI_OPTNAME(opt) "-" #opt
#define OPTNAME(opt)      "--" #opt
#define OPTNAME_V(opt)    "V--" #opt "="
#define OPTION(opt)       OPT_##opt
#define INTERNAL          "__internal__"

typedef enum {
	OPTION(unknown) = -1, /* unknown option */
	
	OPTION(dummy1),   /* Info/help options */
	OPTION(help),     OPTION(verbose),   OPTION(internal),
	OPTION(devinfo),  OPTION(devvinfo),  OPTION(check),    
	OPTION(rdevaction), OPTION(bundlekernels),
	
	OPTION(lastoption)    /* marker */
} option_t;

char *optnames[] = {
	NULL,

	OPTNAME(help),     OPTNAME(verbose),   INTERNAL,
	OPTNAME(devinfo),  OPTNAME(devvinfo),  OPTNAME_V(check),  
	OPTNAME_V(rdev-action), OPTNAME(bundle-kernels),
	
	NULL
};


option_t optid(char *arg, char **val)
{
	int i;

	for (i = 0; i < OPTION(lastoption); i++)
	{
		if (!optnames[i])   /* Skip dummy options */
			continue;
		if (optnames[i][0] == 'V')     /* Option with value */
		{
			if (strncmp(optnames[i]+1, arg, strlen(optnames[i])-1) == 0)
			{
				*val = arg + strlen(optnames[i]) - 1;
				return ((option_t) i);
			}
		}
		else
			if (strcmp(optnames[i], arg) == 0)
				return ((option_t) i);
	}
	return (OPTION(unknown));
}


#ifdef OMPI_REMOTE_OFFLOADING

/* Handle string given after "--rdev-action" argument.
 * Currently actions are hard-coded...
 */
static void handle_rdev_action(char *action)
{
	/* --rdev-action="get-num-mpiprocs" */
	if (strcmp(action, "get-num-mpiprocs") == 0)
	{
		getnprocs = true;
		return;
	}
	
	/* --rdev-action="get-mpi-hosts" */
	if (strcmp(action, "get-mpi-hosts") == 0)
	{
		getmpihosts = true;
		return;
	}
	
	/* --rdev-action="create-mpirun-script" */
	if (strcmp(action, "create-mpirun-script") == 0)
	{
		creatempirunscript = true;
		return;
	}
	
	fprintf(stderr, "Unknown remote devices action '%s'.\n", action);
	exit(0);
}

#endif


void parse_args(int argc, char **argv)
{
	int  d, oid;
	char *parameter, *val, *ompiconf_bin = argv[0];

	argv++;
	argc--;

	while (argc)
	{
		switch (oid = optid(parameter = argv[0], &val))
		{
			case OPTION(help):
				ompi_info();
				ompi_help(ompiconf_bin);
				exit(0);
				break;
			case OPTION(verbose):
				goto BEVERB;
			case OPTION(internal):  
				internal = true; 
				break;
			case OPTION(devinfo):   
				showdevinfo = true; 
				break;
			case OPTION(devvinfo):  
				showdevinfo = true; 
				verbose = true; 
				break;
#if defined(ENABLE_KERNEL_BUNDLING)
			case OPTION(bundlekernels):     
				argc--; argv++;
				ompiconf_bundle_kernel_files(&argc, &argv);
				return;
#endif
			case OPTION(check):     
				if (!internal) goto UNSUPPORTED;
				module_query = strndup(val, strlen(val)); 
				checkmodule = true; 
				break;
#ifdef OMPI_REMOTE_OFFLOADING
			case OPTION(rdevaction): 
				if (!internal) goto UNSUPPORTED;
				handle_rdev_action(val);
				break;
#endif
			default:
				if (parameter[0] == '-')
				{
					switch (parameter[1])
					{
						case 'v':
							BEVERB:
							verbose = true;
							break;
						default:
							goto UNSUPPORTED;
					}
				} else {
					UNSUPPORTED:
					fprintf(stderr, "Command not supported, run %s with no arguments for help.\n", 
									 ompiconf_bin);
					exit(0);
				}
		}
		
		argc = argc - 1;
		argv = argv + 1;
	}
}

int main(int argc, char *argv[])
{
	int check_failed;

	parse_args(argc, argv);
	ompiconf_modules_employ(NULL);
	
#ifdef OMPI_REMOTE_OFFLOADING
	ompiconf_remote_modules_employ();
#endif

	if (argc == 1)
	{
		ompi_info();
		ompi_help(argv[0]);
		exit(0);
	}

	if (showdevinfo)
	{
		fprintf(stderr, "%d configured device module(s): %s\n\n",
				oc_nmodules, MODULES_CONFIG);
		ompiconf_modules_show_info(verbose);
#ifdef OMPI_REMOTE_OFFLOADING
		ompiconf_remote_modules_finalize();
#endif
		exit(0);
	}

	if (internal)
	{
#ifdef OMPI_REMOTE_OFFLOADING
		if (getnprocs) /* rdev query */
		{
			printf("%d\n", ompiconf_get_num_mpiprocs() + 1); /* master + workers */
			exit(0);
		}
		
		if (getmpihosts)
		{
			printf("%s\n", ompiconf_get_node_names());
			exit(0);
		}
		
		if (creatempirunscript)
		{
			ompiconf_create_mpirun_script();
			exit(0);
		}
#endif
		
		if (checkmodule)
		{
			check_failed = ompiconf_modules_query(module_query);
			free(module_query);
			exit(check_failed);
		}
	}

#ifdef OMPI_REMOTE_OFFLOADING
	ompiconf_remote_modules_finalize();
#endif
	
}