/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_combine.c -- statement combinations and splits.
 * 
 * 
 * Combination/Split rules
 * -----------------------
 *
 * Combination rules describe pairs of directives that can be combined into
 * a single directive. Split rules describe which two directives should
 * a combined directive be split into. We organize them into 
 * arbitraty combination rulesets, as above.
 * 
 * To add a combination rule, simply list the outer, the inner and 
 * the combined directive names, as in 	{ DCTEAMS, DCDISTRIBUTE, DCTEAMSDIST }.
 * At this point we do not care if the resulting construct is a combined
 * or a composite one.
 *
 * To add a split rule, simply list the combined directive name, as well
 * as the produced outer and inner directive names, as in 
 * { DCTARGETTEAMSDIST, DCTARGETTEAMS, DCDISTRIBUTE }.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "x_combine.h"
#include "x_clauserules.h"
#include "x_clauses.h"
#include "set.h"
#include "ast_copy.h"
#include "ast_free.h"

#define COMBSET_DEFINE(m) combrule_t *m ## _mergeinfo = (combrule_t[])
#define COMBSET(m)        &m ## _mergeinfo
#define COMBSEND          { DCNONE, DCNONE, DCNONE }

#define SPLITSET_DEFINE(m) splitrule_t *m ## _splitinfo = (splitrule_t[])
#define SPLITSET(m)        &m ## _splitinfo
#define SPLITSEND          { DCNONE, DCNONE, DCNONE }

typedef struct ompdir_merge_rule
{
	ompdirt_e outer;
	ompdirt_e inner;
	ompdirt_e result;
} combrule_t;

typedef struct ompdir_split_rule
{
	ompdirt_e combined;
	ompdirt_e outer;
	ompdirt_e inner;
} splitrule_t;


/************************************************
 *                                              *
 *                 COMBINATIONS                 *
 *                                              *
 ************************************************/


COMBSET_DEFINE(distribute) 
{
	{ DCDISTRIBUTE, DCPARFOR, DCDISTPARFOR },   /* composite */
	COMBSEND
};

COMBSET_DEFINE(teams) 
{
	{ DCTEAMS, DCDISTRIBUTE, DCTEAMSDIST },
	{ DCTEAMS, DCDISTPARFOR, DCTEAMSDISTPARFOR },
	COMBSEND
};

COMBSET_DEFINE(target) 
{
	{ DCTARGET, DCTEAMS, DCTARGETTEAMS },
	{ DCTARGET, DCTEAMSDIST, DCTARGETTEAMSDIST },
	{ DCTARGET, DCTEAMSDISTPARFOR, DCTARGETTEAMSDISTPARFOR },
	{ DCTARGETTEAMS, DCDISTPARFOR, DCTARGETTEAMSDISTPARFOR },
	{ DCTARGETTEAMS, DCDISTRIBUTE, DCTARGETTEAMSDIST },
	{ DCTARGET, DCPARALLEL, DCTARGETPARALLEL },
	{ DCTARGET, DCPARFOR, DCTARGETPARFOR },
	COMBSEND
};

combrule_t **all_combrules[] = 
{ 
	COMBSET(target), COMBSET(teams), COMBSET(distribute),
	NULL
};


/************************************************
 *                                              *
 *                   SPLITS                     *
 *                                              *
 ************************************************/


SPLITSET_DEFINE(target) 
{
	{ DCTARGETTEAMSDIST, DCTARGETTEAMS, DCDISTRIBUTE },
	{ DCTARGETTEAMSDISTPARFOR, DCTARGETTEAMS, DCDISTPARFOR },
	{ DCTARGETTEAMS, DCTARGET, DCTEAMS },
	{ DCTARGETPARFOR, DCTARGET, DCPARFOR },
	{ DCTARGETPARALLEL, DCTARGET, DCPARALLEL },
	COMBSEND
};

SPLITSET_DEFINE(parallel) 
{
	{ DCPARSECTIONS, DCPARALLEL, DCSECTIONS },
	{ DCPARFOR, DCPARALLEL, DCFOR_P },
	COMBSEND
};

SPLITSET_DEFINE(teams) 
{
	{ DCTEAMSDIST, DCTEAMS, DCDISTRIBUTE },
	{ DCTEAMSDISTPARFOR, DCTEAMS, DCDISTPARFOR },
	COMBSEND
};


splitrule_t **all_splitrules[] = 
{ 
	SPLITSET(target), SPLITSET(parallel), SPLITSET(teams),
	NULL
};


/**
 * Get corresponding combined version of two given OMP statements
 *
 * @param  t     The outer OMP statement
 * @param  with  The inner OMP statement
 * @return The directive type of the combined construct, otherwise DCNONE.
 */
static
ompdirt_e ccc_combtype(aststmt t, aststmt with) 
{
	ompdirt_e ttype, withtype, outer;
	int i, j;

	if ((t->type != OMPSTMT) || (with->type != OMPSTMT))
		return DCNONE;

	ttype = OmpStmtDir(t)->type;
	withtype = OmpStmtDir(with)->type;
	for (i = 0; all_combrules[i] != NULL; i++)
		for (j = 0; (outer = (*all_combrules[i])[j].outer) != DCNONE; j++)
			if ((outer == ttype) && ((*all_combrules[i])[j].inner == withtype))
				return (*all_combrules[i])[j].result;
		
	return DCNONE;
}

/**
 * Check if one OMP statement can be combined with another, according to
 * the combination rules
 *
 * @param  t The OMP statement
 * @return true if given OMP statement can be combined with another one.
 */
bool ccc_iscombinable(aststmt t) 
{
	ompdirt_e mtype, outer, inner;
	aststmt innerstmt;
	int i, j;

	if (t->type != OMPSTMT)
		return false;

	innerstmt = t->u.omp->body; 
	if (innerstmt->type == COMPOUND)   /* A compound statement */
		innerstmt = innerstmt->body;   /* Should contain only 1 OpenMP statement */

	if (ccc_combtype(t, innerstmt) == DCNONE)
		return false;

	mtype = OmpStmtDir(t)->type;
	for (i = 0; all_combrules[i] != NULL; i++)
	{
		for (j = 0; (outer = (*all_combrules[i])[j].outer) != DCNONE; j++)
			if (outer == mtype)
				return true;
		for (j = 0; (inner = (*all_combrules[i])[j].inner) != DCNONE; j++)
			if (inner == mtype)
				return true;
	}

	return false;
}


/**
 * Produce the combined version of two directives. The result 
 * replaces the first directive.
 *
 * @param t1      The parent (outer) directive that encloses t2
 * @param t2      The child (inner) directive
 * @param newtype The resulting directive type
 */
static
void ccc_combine_ompstmts(aststmt *t1, aststmt *t2, ompdirt_e newtype)
{
	ompclause newcls = NULL;

	/* The resulted is stored in the first stmt */
	(*t1)->u.omp->type = newtype;  
	OmpStmtDir(*t1)->type = newtype;

	/* Merge the clauses of the two directives */
	xc_merge_clauses(&newcls, OmpStmtDir(*t1)->clauses, OmpStmtDir(*t2)->clauses);
	OmpStmtDir(*t1)->clauses = ast_ompclause_copy(newcls);

	/* Some bookkeeping... */
	(*t1)->u.omp->body->file =            
		(*t2)->u.omp->file =
			OmpStmtDir(*t2)->file =
				OmpStmtDir(*t1)->file;
	(*t1)->u.omp->body->l =
		(*t2)->u.omp->l =
			OmpStmtDir(*t2)->l =
				OmpStmtDir(*t1)->l;
	(*t1)->u.omp->body->c =
		(*t2)->u.omp->c =
			OmpStmtDir(*t2)->c =
				OmpStmtDir(*t1)->c;

	/* Store the actual body in the result */
	(*t1)->u.omp->body = (*t2)->u.omp->body;
	ast_stmt_parent((*t1)->parent, (*t1));
}


/**
 * Produce the splitted version of a combined directive. The result 
 * replaces the directive with one construct that encloses another.
 *
 * @param c     The combined directive
 * @param type1 The resulting type of the outer directive
 * @param type2 The resulting type of the inner directive
 */
static
void ccc_split_ompstmts(aststmt *t, ompdirt_e type1, ompdirt_e type2)
{
	ompclause ocls = NULL;
	ompclause icls = NULL;

	if (OmpStmtDir(*t)->clauses)
	{
		if ((*t)->u.omp->type == DCPARFOR || (*t)->u.omp->type == DCPARSECTIONS)
			xc_split_combined_parallel_clauses(OmpStmtDir(*t)->clauses, &ocls, &icls);
		else
			xc_split_clauses_fromdirs(OmpStmtDir(*t)->clauses, type1, type2, &ocls, &icls);
	}

	ast_ompclause_free(OmpStmtDir(*t)->clauses);   /* Not needed */
	(*t)->u.omp->type = type1;                   /* Change the node */
	OmpStmtDir(*t)->type = type1;
	OmpStmtDir(*t)->clauses = ocls;
	(*t)->u.omp->body = OmpStmt(                      /* New body */
							OmpConstruct(
							type2,
							OmpDirective(type2, icls),
							(*t)->u.omp->body
							)
						);
	(*t)->u.omp->body->file =                    /* Fix infos */
		(*t)->u.omp->body->u.omp->file =
		(*t)->u.omp->body->u.omp->directive->file =
			OmpStmtDir(*t)->file;
	(*t)->u.omp->body->l =
		(*t)->u.omp->body->u.omp->l =
		(*t)->u.omp->body->u.omp->directive->l =
			OmpStmtDir(*t)->l;
	(*t)->u.omp->body->c =
		(*t)->u.omp->body->u.omp->c =
		(*t)->u.omp->body->u.omp->directive->c =
			OmpStmtDir(*t)->c;
	ast_stmt_parent((*t)->parent, (*t));   /* Reparentize correctly */
}


/**
 * Try to combine a directive with its enclosed one, according to the
 * combination rules. The combination will take place only if the 
 * enclosed (inner) directive is the only child of the node, either 
 * contained in a compound or not.
 *
 * @param  t     the given directive that may enclose another directive
 * @return true  if a combined statement was indeed possible
 */
bool ccc_try_combining(aststmt *t)
{
	ompdirt_e resulttype;
	aststmt   inner;

	if (!(*t) || !ccc_iscombinable(*t))
		return false;

	inner = (*t)->u.omp->body; 
	if (inner->type == COMPOUND)   /* A compound statement */
		inner = inner->body;   /* Should contain only 1 OpenMP statement */

	resulttype = ccc_combtype((*t), inner);

	ccc_combine_ompstmts(t, &inner, resulttype);
	return true;
}


/**
 * Try to split a directive into two directives, according to the
 * split rules. 
 *
 * @param  t      the given combined directive
 * @return true   if a combined statement was indeed possible
 */
bool ccc_try_splitting(aststmt *t)
{
	ompdirt_e resulttype1, resulttype2;

	if (!ccc_issplittable((*t), &resulttype1, &resulttype2))
		return false;

	ccc_split_ompstmts(t, resulttype1, resulttype2);
	return true;
}


/**
 * Check if an OMP statement can be split to two statements, one nested 
 * within the other, according to the split rules.
 *
 * @param  t The OMP statement
 * @return true if given OMP statement can be split.
 */
bool ccc_issplittable(aststmt t, ompdirt_e *outer, ompdirt_e *inner)
{
	ompdirt_e mtype, combined;
	int i, j;

	if (t->type != OMPSTMT)
		return false;

	mtype = OmpStmtDir(t)->type;
	for (i = 0; all_splitrules[i] != NULL; i++)
		for (j = 0; (combined = (*all_splitrules[i])[j].combined) != DCNONE; j++)
			if (combined == mtype)
			{
				if (outer) *outer = (*all_splitrules[i])[j].outer;
				if (inner) *inner = (*all_splitrules[i])[j].inner;
				return true;
			};
	return false;
}

