/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_xformrules.c -- transformation rules */

/* This unit registers transformation functions for each OpenMP construct.
 * In addition, it defines the handling policy (split/combine) for combined,
 * composite and exlusively nested(*) constructs.
 * 
 * (*) We call two constructs A and B *exclusively nested* if B is closely 
 * nested within A and there is no other statement nested in A, e.g.
 * #pragma omp parallel            // A
 * {
 *   #pragma omp for               // B
 *   for (i=0;...) { ... }
 * } 
 * 
 * See below how to add customized rules for modules/devices.
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "str.h"
#include "ompi.h"
#include "builder.h"
#include "autoscope.h"
#include "set.h"
#include "ast_xform.h"
#include "ast_xformrules.h"
#include "assorted.h"
#include "ast_copy.h"
#include "x_clauses.h"
#include "x_clauserules.h"
#include "x_combine.h"
#include "x_parallel.h"
#include "x_task.h"
#include "x_single.h"
#include "x_sections.h"
#include "x_loops.h"
#include "x_for.h"
#include "x_thrpriv.h"
#include "x_shglob.h"
#include "x_target.h"
#include "x_kernels.h"
#include "x_decltarg.h"
#include "x_cars.h"
#include "x_teams.h"
#include "x_arrays.h"
#include "x_distribute.h"
#include "x_simd.h"
#include "x_ordered.h"
#include "x_atomic.h"
#include "x_cancel.h"

/* Device-specific headers */
#include "x_target_cu.h"
#include "x_target_ocl.h"
#include "x_parallel_cu.h"
#include "x_parallel_ocl.h"
#include "x_teams_cu.h"
#include "x_teams_ocl.h"
#include "x_task_cu.h"


#define XFR_DEFINE(d)        struct xfr_vars_ d##_xformvars; \
                             xfr_rule_t *d##_xforminfo = (xfr_rule_t[])
#define XFR_DEFINE_COPY(d,c) struct xfr_vars_ d##_xformvars; \
                             xfr_rule_t *d##_xforminfo = (xfr_rule_t[]) \
                               { { DCNONE, (void *) 0, 0, #c } };

#define XFR_MOD(d)           { #d, &d##_xforminfo, &d##_xformvars, XFR_MODID_ ## d }
#define XFR_LASTMOD          { "", (xfr_rule_t**) 0, (xfr_vars_t) 0, -1 }
#define XFR_LASTRULE         { DCNONE, (void*) 0, 0 }
#define XFR_DEFINE_DEFAULT   XFR_DEFINE(_)
#define DEFAULT_XFR_MOD      XFR_MOD(_)

#define XFR_GET(x)           (all_xformrules[x])
#define XFR_RULES(x)         (XFR_GET(x).rules)
#define XFR_DEVNAME(x)       (XFR_GET(x).modulename)
#define XFR_VARS(x)          (XFR_GET(x).vars)
#define XFR_ID(x)            (XFR_GET(x).id)
#define XFR_GETVAR(x,v)      (XFR_VARS(x)->v)

#define XFR_IS(x,y)          (!strcmp(XFR_DEVNAME(x), y))
#define XFR_ISLAST(x)        ((!strcmp(XFR_DEVNAME(x), "")) && \
                              (XFR_RULES(x) == NULL))

#define XFR_GETRULE(x,y)     ((*XFR_RULES(x))[y])
#define XFR_GETRULETYPE(x,y) (XFR_GETRULE(x,y).dirtype)

#define XFR_ISLASTRULE(x,y)  (XFR_GETRULETYPE(x,y) == DCNONE)
#define XFR_ISCOPY(x,y)      ((XFR_GETRULETYPE(x,y) == DCNONE) && \
                              (XFR_GETCOPY(x,y) != NULL))
                              
xfr_moduleid_e ast_xfr_modulename_toid(char *modulename)
{
	int i;
	for (i = 0; all_xformrules[i].id != -1; i++)
		if (!strncmp(all_xformrules[i].modulename, modulename, 
		             strlen(modulename)))
		    return all_xformrules[i].id;
	return XFRMOD(unknown);
}

char *ast_xfr_moduleid_toname(xfr_moduleid_e id)
{
	int i;
	for (i = 0; all_xformrules[i].id != -1; i++)
		if (all_xformrules[i].id == id)
		    return all_xformrules[i].modulename;
	return NULL;
}


/* The DEFAULT rules apply to the host and, in absence of other rules,
 * to the code targeting all modules/devices.
 * 
 * To define new, module-specific transformation rules for some constructs,
 * use the following rules syntax:
 * 
 * XFR_DEFINE(<modulename>) {
 *     { <constuct-type>, <xform-function>, <combine/split-policy> },
 *     ...,
 *     XFR_LASTRULE       // This ends the new transformation rules
 * }
 * 
 * Then include the new transformation rules to the global collection:
 * 
 * xfr_mod_t all_xformrules[] = {
 *     DEFAULT_XFR_MOD,   // Should always be first
 *     ...,
 *     XFR_MOD(<modulename>),
 *     XFR_LASTMOD        // Should always be last
 * }
 * 
 * Important notes:
 * 
 *   Directives that have been assigned with a policy different
 *   from NONE, will be handled by the transformation function of the
 *   resulted directive.
 *   
 * Examples:
 * (a) #pragma omp target parallel for 
 * 
 *     with SPLITONLY gets transformed to:
 * 
 *     #pragma omp target 
 *       #pragma omp parallel for
 *   
 *     and handled as a `#pragma omp target'
 *     (i.e. the corresponding DCTARGET xform function will be called)
 * 
 * (b) #pragma omp target teams distribute
 *       #pragma omp parallel for
 * 
 *     with BOTH policy gets transformed to:
 * 
 *     (1) #pragma omp target teams distribute parallel for
 *  
 *     then to:
 *  
 *     (2) #pragma omp target teams
 *           #pragma omp distribute parallel for
 *    
 *     and gets handled as a `#pragma omp target teams'
 *     (i.e. the corresponding DCTARGETTEAMS xform function will be called)
 */

XFR_DEFINE_DEFAULT {
	{ DCTARGET,                    xform_target_default,        X_ACTION_NONE },
	{ DCTARGETUPD,                 xform_targetupdate,          X_ACTION_NONE },
	{ DCTARGETDATA,                xform_targetdata,            X_ACTION_NONE },
	{ DCDECLTARGET,                xform_declaretarget,         X_ACTION_NONE },
	{ DCTARGENTERDATA,             xform_targetenterdata,       X_ACTION_NONE },
	{ DCTARGEXITDATA,              xform_targetexitdata,        X_ACTION_NONE },
	{ DCTARGETPARALLEL,            xform_targetparallel,        X_ACTION_SPLIT },
	{ DCTARGETPARFOR,              xform_targparfor_default,    X_ACTION_SPLIT },
	{ DCTARGETTEAMS,               xform_targetteams,           X_ACTION_SPLIT },
	{ DCTARGETTEAMSDIST,           xform_targetteams,           X_ACTION_SPLIT },
	{ DCTARGETTEAMSDISTSIMD,       xform_targetteamsdistsimd,   X_ACTION_NONE },
	{ DCTARGETTEAMSDISTPARFOR,     xform_targetteamsdistparfor, X_ACTION_SPLIT },
	{ DCTARGETTEAMSDISTPARFORSIMD, xform_targetteamsdistparforsimd, X_ACTION_NONE },
	{ DCTEAMS,                     xform_teams,                 X_ACTION_NONE },
	{ DCTEAMSDISTSIMD,             xform_teamsdistsimd,         X_ACTION_NONE },
	{ DCTEAMSDISTPARFOR,           xform_teamsdistparfor,       X_ACTION_SPLIT },
	{ DCTEAMSDISTPARFORSIMD,       xform_teamsdistparforsimd,   X_ACTION_NONE },
	{ DCPARALLEL,                  xform_parallel_wrapper,      X_ACTION_NONE },
	{ DCPARSECTIONS,               xform_parallel_wrapper,      X_ACTION_SPLIT },
	{ DCPARFOR,                    xform_parallel_wrapper,      X_ACTION_SPLIT },
	{ DCPARFORSIMD,                xform_parforsimd,            X_ACTION_NONE },
	{ DCDISTRIBUTE,                xform_distribute,            X_ACTION_NONE },
	{ DCDISTSIMD,                  xform_distsimd,              X_ACTION_NONE },
	{ DCDISTPARFOR,                xform_distparfor,            X_ACTION_NONE },
	{ DCDISTPARFORSIMD,            xform_distparforsimd,        X_ACTION_NONE },
	{ DCFORSIMD,                   xform_forsimd,               X_ACTION_NONE },
	{ DCSIMD,                      xform_simd,                  X_ACTION_NONE },
	{ DCCANCEL,                    xform_cancel,                X_ACTION_NONE },
	{ DCCANCELLATIONPOINT,         xform_cancellationpoint,     X_ACTION_NONE },
	{ DCTHREADPRIVATE,             xform_threadprivate,         X_ACTION_NONE },  
	{ DCATOMIC,                    xform_atomic,                X_ACTION_NONE },
	{ DCBARRIER,                   xform_barrier,               X_ACTION_NONE },
	{ DCTASKWAIT,                  xform_taskwait,              X_ACTION_NONE },
	{ DCTASKYIELD,                 xform_taskyield,             X_ACTION_NONE },
	{ DCFLUSH,                     xform_flush,                 X_ACTION_NONE },
	{ DCCRITICAL,                  xform_critical,              X_ACTION_NONE },
	{ DCMASTER,                    xform_master,                X_ACTION_NONE },
	{ DCMASKED,                    xform_masked,                X_ACTION_NONE },
	{ DCORDERED,                   xform_ordered,               X_ACTION_NONE },
	{ DCSINGLE,                    xform_single,                X_ACTION_NONE },
	{ DCSECTION,                   xform_section,               X_ACTION_NONE },
	{ DCSECTIONS,                  xform_sections,              X_ACTION_NONE },
	{ DCFOR_P,                     xform_for,                   X_ACTION_NONE },
	{ DCFOR,                       xform_for,                   X_ACTION_NONE },
	{ DCTASK,                      xform_task,                  X_ACTION_NONE },
	{ DCTASKGROUP,                 xform_taskgroup,             X_ACTION_NONE },
	XFR_LASTRULE /* new-directive.sh:xfrules */
};

XFR_DEFINE(opencl) {
	{ DCTARGET,                xform_target_opencl,             X_ACTION_COMBINE },
	{ DCTARGETPARALLEL,        xform_targetparallel_opencl,     X_ACTION_SPLIT },
	{ DCTARGETPARFOR,          xform_targparfor_opencl,         X_ACTION_SPLIT },
	{ DCTARGETTEAMS,           xform_targetteams_opencl,        X_ACTION_NONE },
	{ DCTARGETTEAMSDIST,       xform_targetteams_opencl,        X_ACTION_COMBSPLIT },
	{ DCTARGETTEAMSDISTPARFOR, xform_targetteamsdistparfor_opencl,   X_ACTION_SPLIT },
	{ DCPARFOR,                xform_parallel_wrapper,          X_ACTION_SPLIT },
	{ DCPARALLEL,              xform_parallel_opencl_wrapper,   X_ACTION_NONE },
	XFR_LASTRULE
};

/* Transformation rules for CUDA devices */
XFR_DEFINE(cuda) {
	{ DCTARGET,                xform_target_cuda,             X_ACTION_COMBINE },
	{ DCTARGETPARALLEL,        xform_targetparallel_cuda,     X_ACTION_SPLIT },
	{ DCTARGETPARFOR,          xform_targparfor_cuda,         X_ACTION_SPLIT },
	{ DCTARGETTEAMS,           xform_targetteams_cuda,        X_ACTION_NONE },
	{ DCTARGETTEAMSDIST,       xform_targetteams_cuda,        X_ACTION_COMBSPLIT },
	{ DCTARGETTEAMSDISTPARFOR, xform_targetteamsdistparfor_cuda,   X_ACTION_SPLIT },
	{ DCCRITICAL,              xform_critical_cuda,           X_ACTION_NONE },
	{ DCATOMIC,                xform_atomic_cuda,             X_ACTION_NONE },
	{ DCPARFOR,                xform_parallel_wrapper,        X_ACTION_SPLIT },
	{ DCPARALLEL,              xform_parallel_cuda_wrapper,   X_ACTION_NONE },
	{ DCTASK,                  xform_task_cuda,        X_ACTION_NONE },
	XFR_LASTRULE
};

/* new-module.sh:ruleset */

/* The full rules collection */
xfr_mod_t all_xformrules[] = { 
	DEFAULT_XFR_MOD, 
	XFR_MOD(opencl),
	XFR_MOD(cuda),
	XFR_LASTMOD /* new-module.sh:allrules */
};

/* Set keeping all the transformation rules. */
SET_TYPE_IMPLEMENT(xformrules)
set(xformrules) ast_xfrules;

/* Boolean set to `true' only when rules are successfully
 * parsed. 
 */
static bool ast_xfr_inited = false;

/**
 * Call the transformation function for a specific
 * directive, according to the transformation rules of
 * a specific module.
 * 
 * @param modulename The module name
 * @param t          The input tree
 */
void ast_xfr_call_xformfunc(xfr_moduleid_e modid, aststmt *t)
{
	setelem(xformrules) e, defxfrules = ast_xfr_get_defaultrules();

	if (!ast_xfr_inited)
		exit_error(1, "[%s] xform rules not initialized; exiting.\n",
		"ast_xfr_call_xformfunc");

	assert((e = set_get(ast_xfrules, modid)) != NULL);
	if (!e->value->rules[(*t)->u.omp->type].initialized)
		assert(defxfrules->value->rules[(*t)->u.omp->type].initialized);

	if (!e->value->rules[(*t)->u.omp->type].xform_func)
		defxfrules->value->rules[(*t)->u.omp->type].xform_func(t);
	else
		e->value->rules[(*t)->u.omp->type].xform_func(t);
}

/**
 * Parse the declared transformation rules.
 */
void ast_xfr_init_rules(void)
{
	int i, j, realindex;
	setelem(xformrules) e;

	set_init(xformrules, &ast_xfrules);
	for (i = 0; !XFR_ISLAST(i); i++)
	{
		/* If the ruleset is a copy of another one, then
		 * use the index of the original set */
		realindex = i;
		e = set_put(ast_xfrules, all_xformrules[i].id);
		e->value = (xfr_modrules_t *) malloc(sizeof(xfr_modrules_t));
		e->value->origindex = i;
		e->value->vars = XFR_VARS(i);
		e->value->vars->newglobals = e->value->vars->targtree = NULL;
		e->value->vars->targetnum = 0;
		e->value->vars->Targets = NULL;
		for (j = 0; !XFR_ISLASTRULE(realindex, j); j++)
		{
			e->value->rules[XFR_GETRULETYPE(realindex, j)].initialized = true;
			e->value->rules[XFR_GETRULETYPE(realindex, j)].xform_func = 
				XFR_GETRULE(realindex, j).xform_func;
			e->value->rules[XFR_GETRULETYPE(realindex, j)].combsplittype = 
				XFR_GETRULE(realindex, j).combsplittype;
		}
	}

	ast_xfr_inited = true;
}

/**
 * Get the default transformation rules
 */
setelem(xformrules) ast_xfr_get_defaultrules(void)
{
	return set_get(ast_xfrules, XFRMOD(_));
}

/**
 * Returns `true', if a directive type is supported
 * by the default transformation rules.
 * 
 * @param dirtype The directive type
 */
bool ast_xfr_type_issupported(ompdirt_e dirtype)
{
	setelem(xformrules) defxfrules = ast_xfr_get_defaultrules();
	return defxfrules->value->rules[dirtype].initialized;
}

/**
 * Generate N copies of an input tree, where N = the number
 * of different modules that have customized transformation
 * rules.
 *
 * @param t The input tree
 */
void ast_xfr_gen_tree_copies(aststmt *t)
{
	setelem(xformrules) e, defxfrules = ast_xfr_get_defaultrules();
	defxfrules->value->vars->tree = t;

	for (e = ast_xfrules->first; e; e = e->next)
	{
		if (e == defxfrules)
			continue;

		if (!is_substr(str_string(modstr), ast_xfr_moduleid_toname(e->key)))
			continue;

		if (!e->value->rules[(*t)->u.omp->type].initialized)
			continue;

		e->value->vars->tree = (aststmt *) malloc(sizeof(aststmt));
		assert(e->value->vars->tree != NULL);
		*(e->value->vars->tree) = ast_stmt_copy(*t);
		ast_stmt_parent((*t)->parent, *(e->value->vars->tree));
	}	
}

/**
 * Free all the generated tree copies for each set of
 * transformation rules
 */
void ast_xfr_free_tree_copies(void)
{
	setelem(xformrules) e, defxfrules = ast_xfr_get_defaultrules();

	for (e = ast_xfrules->first; e; e = e->next)
	{
		if (e == defxfrules || !is_substr(str_string(modstr), ast_xfr_moduleid_toname(e->key)))
			continue;
		free(e->value->vars->tree);
	}	
}
