/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_assorted.c - Assorted functions and utilities related to the AST */

#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include <string.h>
#include "ast_copy.h"
#include "ast_show.h"
#include "ast_assorted.h"
#include "ast_print.h"
#include "ompi.h"
#include "str.h"


/**
 * @brief Given the u.decl of a struct/union, find the given field declaration
 * 
 * @param d  the u.decl of the spec of a struct or union
 * @param f  the field symbol we need to find
 * @return the field declaration
 */
astdecl _sufield(astdecl d, symbol f)
{
	astdecl r;
	
	if (d == NULL) return NULL;
	if (d->type == DLIST && d->subtype == DECL_fieldlist)
	{
		if ((r = _sufield(d->u.next, f)) != NULL)
			return r;
		return _sufield(d->decl, f);
	}
	else    /* Should be just a DSTRUCTFIELD */
	{
		astdecl r;

		if (d->type != DSTRUCTFIELD) return NULL;
		r = decl_getidentifier(d->decl);
		return (r && r->u.id == f) ? d : NULL;
	}
}


/**
 * @brief Get the filed of a struct or union
 * 
 * @param s the name of the struct or union
 * @param f the name of the field
 * @return the structfield or NULL if s or f are unknown
 */
astdecl get_sufield(symbol s, symbol f)
{
	stentry e = symtab_get(stab, s, SUNAME);
	astspec sp;
	
	/* Sanity checks that the struct or union exists */
	if (!e) return NULL;
	if (!(sp = e->spec)) return NULL;
	if ((sp = speclist_getspec(sp, SUE, 0)) == NULL) return NULL;
	if (sp->subtype != SPEC_struct && sp->subtype == SPEC_union) return NULL;
	
	return _sufield(sp->u.decl, f);
}


aststmt linepragma(int line, symbol file)
{
	return ((cppLineNo) ? verbit("# %d \"%s\"", line, file->name) : verbit(""));
}


/* This simply prints the directive into a string and encloses it
 * in comments, adding also the source line number.
 * It returns a verbatim node with the comment.
 */
aststmt ompdir_commented(ompdir d)
{
	static str bf = NULL;
	aststmt    st;

	if (bf == NULL) bf = Strnew();
	str_printf(bf, "/* (l%d) ", d->l);
	ast_ompdir_print(bf, d);
	str_seek(bf, str_tell(bf)-1);      /* get rid of the \n */
	str_printf(bf, " */");

	st = Verbatim(strdup(str_string(bf)));
	if (cppLineNo) 
	  st = BlockList(st, linepragma(d->l, d->file));
	  
	str_truncate(bf);
	return (st);
}
