/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* remotedev/workers.c
 * 
 * All worker (remote; with rank != 0) MPI processes execute
 * these functions. Their purpose is to respond to the
 * primary node's requests. Note that the primary and the worker nodes do
 * **NOT** run in a shared memory environment. Communication is
 * achieved with MPI.
 */

// #define DBGPRN_FORCE 
// #define DBGPRN_BLOCK
#define DBGPRN_FILTER DBG_RDEV_WORKERS

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include "ort_prive.h"
#include "../../rt_common.h"

#ifdef OMPI_REMOTE_OFFLOADING
#include "remotedev/workers.h"
#include "remotedev/workercmds.h"
#include "remotedev/memory.h"
#include "remotedev/node_manager.h"
#include "remotedev/rdev_prive.h"
#include "rdev_config.h"

#define _workercmd_exec(M) \
	(worker_commands[M.content[0]])(M.tag, M.content)
#define _workercmd_is_valid(command) \
	((command.content[0] >= 0) && (command.content[0] < CMD(LAST)))

static volatile ee_lock_t rdev_lock;
static void **local_devinfos; /* Used by workers to store their local devinfos */


/* Made weak so that if found at load time they get overridden.
 * We need (weak) definitions to cover the case where the module is only 
 * loaded for queries e.g. by ompicc --devinfo; in such cases ORT is not 
 * present and those symbols would otherwise be undefined.
 */
#pragma weak actual__dev_med2dev_addr
char *(*actual__dev_med2dev_addr)(void *, unsigned long);
#pragma weak actual_omp_is_initial_device
int (*actual_omp_is_initial_device)(void);


char *rdev__dev_med2dev_addr(void *medaddr, unsigned long size)
{
	DBGPRN((stderr, "[remotedev worker %d] _dev_med2dev_addr %p\n", 
	             getpid(), medaddr));
	return medaddr;
}


void override__dev_med2dev_addr(void)
{
	extern char *(*actual__dev_med2dev_addr)(void *, unsigned long);
	actual__dev_med2dev_addr = rdev__dev_med2dev_addr;
}


int rdev_omp_is_initial_device(void)
{
	return 0;
}


void override_omp_is_initial_device(void)
{
	extern int (*actual_omp_is_initial_device)(void);
	actual_omp_is_initial_device = rdev_omp_is_initial_device;
}


/* Initialize all local devices and construct an array
 * containing all devinfos.
 */
static void init_local_devices(void)
{
	int did, i = 0;
	ee_init_lock((ee_lock_t *) &rdev_lock, ORT_LOCK_NORMAL);

	local_devinfos = (void**) smalloc(ort->num_local_devices * sizeof(void *));

	/* Initialize all non-remote devices (excluding hostdev which is already initialized) */
	for (did = 0; did < ort->num_local_devices; did++)
	{
		if (ort->ort_devices[did].module->remote)
			continue;

		DBGPRN((stderr, "[remotedev] WORKER @ %s initializing device %d of module \"%s\"\n",
		             Comm_Get_info(comminfo), did, ort->ort_devices[did].module->name));
						
		ee_set_lock((ee_lock_t *) &rdev_lock);
		if (!ort->ort_devices[did].initialized)
			ort_init_device(did);
		ee_unset_lock((ee_lock_t *) &rdev_lock);
		
		local_devinfos[i++] = ort->ort_devices[did].device_info;
	}
	
	/* Inform the primary node (waiting @ receive_all_devinfos) */
	Comm_Send(comminfo, PRIMARY_NODE, COMM_BYTE, local_devinfos, 
	          ort->num_local_devices * sizeof(void*), 0, NULL);

	free(local_devinfos); /* not needed anymore */
	local_devinfos = NULL;
}


/* Retrieves node entry in rdev config and returns installed modules */
char **rdev_worker_get_modules(int *nModules)
{
	int index;
	
	/* Initialize remotedev configuration from the embedded hex */
#ifdef IGNORE_REMOTE_DEVICES_SNAPSHOT
	rdev_config_initialize(DONT_IGNORE_DISABLED_MODULES);
#else
	rdev_config_initialize_from_hex(ompi_remote_devices, ompi_remote_devices_size, DONT_IGNORE_DISABLED_MODULES);
#endif
	
	/* Learn my ID */
	index = rdev_man_get_my_id() - 1; // 1, 2, 3 -> 0, 1, 2
	
	/* TODO check if this is correct */
	*nModules = rdev_config.nodes[index].num_modules;
	if (rdev_config.nodes[index].has_cpu_module)
		(*nModules)--;
		
	return rdev_config.nodes[index].module_names;
}


static void recv_command(command_t *command)
{
	Comm_Status st;

	if (command->content == NULL)
		return;

	/* Type of command to execute + args */
	Comm_Recv(comminfo, PRIMARY_NODE, COMM_UNSIGNED_LONG_LONG, 
	          command->content, COMM_MAX_NUM_ARGS, COMM_ANY_TAG, &st);

	command->tag = st.info_num;
}


/* This is called by the host and the other nodes upon first
 * initialization.
 */
void rdev_worker_init(void)
{
	Comm_Init_worker(comminfo);
	DBGPRN((stderr, "[remotedev] I'm a WORKER @ %s\n", Comm_Get_info(comminfo)));

	override__dev_med2dev_addr();
	override_omp_is_initial_device();

	DBGPRN((stderr, "[remotedev] I'm a WORKER @ %s -- blocking...\n", 
	                Comm_Get_info(comminfo)));
}


void rdev_worker_loop(void)
{
	command_t cmd;
	cmd.content = smalloc(COMM_MAX_NUM_ARGS * sizeof(rdev_datatype_t)); /* recycled */
	
	/* Workers initialize their local devices and send their local
	 * devinfos to the primary node
	 */
	init_local_devices();

	/* Loop waiting for commands */
	for (recv_command(&cmd); _workercmd_is_valid(cmd); recv_command(&cmd))
		_workercmd_exec(cmd);
		
	DBGPRN((stderr, "[remotedev worker (%d)]: illegal command (tag=%d, cmd=%llu) received\n", 
	                getpid(), cmd.tag, cmd.content[0]));

	free(cmd.content);
	
	exit(2);
} 

#undef DEBUG_WORKERS
#endif /* OMPI_REMOTE_OFFLOADING */
