/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* This file provides an API so both primary and worker nodes can
 * allocate and access memory. Each worker process holds an array of
 * allocated items. The primary node holds a similar array, but no real
 * allocation is happenning (variables already exist at the primary node).
 * This is happenning so primary and worker nodes can know which elements are
 * being used without the need to communicate with each other. The index
 * of the array where an item is stored is the mediary address of that
 * item. This means that when you want to transfer an address from the primary node
 * to a worker (and vice-versa) you must use the mediary address.
 * When you want to actually read/write to that memory from a worker process
 * you must use the allocated mediary address.
 *
 * ***NOTE***: For this to work, you must ensure that both the primary node and
 * the worker allocate and deallocate the items they commonly use in the
 * same order.
 */

#ifndef __REMOTEDEV_MEMORY_H__
#define __REMOTEDEV_MEMORY_H__

#include "stddefs.h"

#ifdef OMPI_REMOTE_OFFLOADING

#include <stddef.h>
#include "remotedev/rdev_prive.h"
#include "remotedev/rdev.h"

#define FIRST_ADDRESS      0
#define SIZE_IGNORE        0
#define MEM_PRIVATE_BANK   0

#define __ALLOCTABLE(device_id) rdev_alloc_tables[device_id]

typedef enum mem_add_alloc_
{
	DONT_ALLOC = 0, 
	DO_ALLOC = 1
} mem_add_alloc_e;

typedef enum mem_remove_free_
{
	DONT_FREE = 0, 
	DO_FREE = 1
} mem_remove_free_e;

extern rdev_alloctab_t **rdev_alloc_tables;


extern void  rdev_alloctab_init(int number_of_devices);
extern void  rdev_alloctab_init_global_vars(int device_id, int is_primary);
extern void *rdev_alloctab_add(rdev_alloctab_t *table, rdev_datatype_t maddr, size_t size, int map_memory);
extern void  rdev_alloctab_add_global(rdev_alloctab_t *table, rdev_datatype_t maddr, void *medaddr);
extern rdev_datatype_t rdev_alloctab_register(rdev_alloctab_t *table, int map_memory);
extern void  rdev_alloctab_unregister(rdev_alloctab_t *table, rdev_datatype_t maddr, int unmap_memory);
extern void  rdev_alloctab_remove(rdev_alloctab_t *table, rdev_datatype_t maddr, int unmap_memory);
extern void *rdev_alloctab_get(rdev_alloctab_t *table, rdev_datatype_t maddr);
extern int   rdev_alloctab_get_item(rdev_alloctab_t *table, void *addr);
extern void  rdev_alloctab_free_all(void);
extern rdev_alloctab_t *rdev_alloctab_new(int device_id);

#endif /* OMPI_REMOTE_OFFLOADING */
#endif /* __REMOTEDEV_MEMORY_H__ */
