/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* cancel.c -- OpenMP cancelation-related routines */

#include <assert.h>
#include "ort_prive.h"


/* This function is called when a cancellation point is reached
 */
int _ort_check_cancel(int type)
{
	ort_eecb_t *me = __MYCB;

	switch (type)
	{
		case 0:
			return check_cancel_parallel(me);
		case 1:
			return check_cancel_taskgroup(me);
		case 2:
			return check_cancel_for(me);
		case 3:
			return check_cancel_sections(me);
	}
	return 0;
}


/* Function for enabling thread cancellation in a parallel region
 */
static void enable_cancel_parallel(ort_eecb_t *me)
{
	if (CANCEL_ENABLED())
		if (me->parent != NULL)
			TEAMINFO(me)->cancel_par_active = true;
}


/* Function for enabling thread cancellation in a section region
 */
static void enable_cancel_sections(ort_eecb_t *me)
{
	if (CANCEL_ENABLED())
	{
		if (me->parent != NULL)
			TEAMINFO(me)->cancel_sec_active = true;
		else
			me->cancel_sec_me = true;
	}
}

/* Function for enabling thread cancellation in a for region
 */
static void enable_cancel_for(ort_eecb_t *me)
{
	if (CANCEL_ENABLED())
	{
		if (me->parent != NULL)
			TEAMINFO(me)->cancel_for_active = true;
		else
			me->cancel_for_me = true;
	}
}


/* Function for enabling thread cancellation in a taskgroup region
 */
static void enable_cancel_taskgroup(ort_eecb_t  *me)
{
	if (__CURRTASK(me)->taskgroup != NULL && CANCEL_ENABLED())
		__CURRTASK(me)->taskgroup->is_canceled = true;
}


/**
 * This function is called when #pragma omp cancel is reached
 * @param type the type of cancelled construct
 * @return true if I must be cancelled immediately (after consulting
 *              the environmental variable).
 */
int _ort_enable_cancel(int type)
{
	ort_eecb_t *me = __MYCB;

	switch (type)
	{
		case 0:
			enable_cancel_parallel(me);
			return check_cancel_parallel(me);
		case 1:
			enable_cancel_taskgroup(me);
			return check_cancel_taskgroup(me);
		case 2:
			enable_cancel_for(me);
			if (me->parent)
				return ( check_cancel_for(me) );
			else
			{
				if (check_cancel_for(me))
				{
					me->cancel_for_me = 0;
					return 1;
				}
				else
					return 0;
			}
		case 3:
			enable_cancel_sections(me);
			if (me->parent)
				return ( check_cancel_sections(me) );
			else
			{
				if (check_cancel_sections(me))
				{
					me->cancel_sec_me = 0;
					return 1;
				}
				else
					return 0;
			}
		default:
			assert(false);   /* Should not happen */
			return -1;
	}
}
