/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* KERNELS.C
 * Module and kernels support for ompicc
 */

/* 
 * May 2019:
 *   Created out of code in ompicc.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <errno.h>
#include <libgen.h>
#include "config.h"
#include "git_version.h"
#include "str.h"
#include "ompicc.h"
#include "mapper.h"
#include "assorted.h"
#include "set.h"
#ifdef OMPI_REMOTE_OFFLOADING
	#include "rdev_config.h"
#endif

static
char current_file[PATHSIZE], cwd[PATHSIZE];  /* Current file and working dir */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        MODULES                                            *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


char **modulenames;  /* Pointers to module names */
int  nmodules;

typedef struct modinfo_
{
	char *modulename;
	char *sourcetype;
	char *suffix;
	char *outtype;
	char *outsuffix;
} modinfo_t;

modinfo_t modinfos[] = {
#ifdef CUDA_JIT_COMPILATION
	{ "cuda", "KERNELTYPE_CUDA_C", "-cuda.c", "KERNELTYPE_CUDA_PTX", "-cuda.ptx" },
#else
	{ "cuda", "KERNELTYPE_CUDA_C", "-cuda.c", "KERNELTYPE_CUDA_FATBIN", "-cuda.fatbin" },
#endif
	{ "proc2", "KERNELTYPE_C", ".c", "KERNELTYPE_GENERIC_OUT", "-proc2.out" },
	{ "proc2l", NULL, NULL, NULL, NULL },
	{ "epiphany3", "KERNELTYPE_C", ".c", "KERNELTYPE_SREC", "-epiphany3.srec" },
	{ "epiphany4", "KERNELTYPE_C", ".c", "KERNELTYPE_SREC", "-epiphany4.srec" },
	{ "opencl", "KERNELTYPE_OPENCL_C", "-opencl.c", "KERNELTYPE_OPENCL_OUT", "-opencl.out" },
	/* new-module.sh:kerneltypes */
	{ NULL, NULL, NULL, NULL, NULL }
};

static modinfo_t *get_module_info(char *modulename)
{
	int i;
	for (i = 0; modinfos[i].modulename != NULL; i++)
		if (!strcmp(modinfos[i].modulename, modulename))
			return &(modinfos[i]);

	return NULL;
}


static int is_module_supported(char *modulename)
{
	int i;

	for (i = 0; i < nmodules; i++)
		if (!strcmp(modulenames[i], modulename))
			return 1;
	
	return 0;
}


void modules_employ(char *modstring)
{
	char *s;
	int  i;

	/* Pass 1: find # modules */
	if (!modstring)
		modstring = strdup(MODULES_CONFIG);
	for (; isspace(*modstring) || *modstring == ','; modstring++)
		;
	if (*modstring == 0)
		return;
	for (nmodules = 1, s = modstring; *s; s++)
	{
		if (isspace(*s) || *s == ',')
		{
			for (*s = 0, s++; isspace(*s) || *s == ','; s++)
				*s = ',';  /* all spaces become commas */
			if (*s)
				nmodules++;
			s--;
		}
	}

	/* Pass 2: fix pointers */
	if ((modulenames = (char **) malloc(nmodules*sizeof(char *))) == NULL)
	{
		fprintf(stderr, "cannot allocate memory");
		exit (1);
	}
	for (i = 0, s = modstring; i < nmodules; i++)
	{
		for (modulenames[i] = s++; *s; s++)
			;
		if (i == nmodules-1)
			break;
		for (; *s == 0 || *s == ','; s++)
			;
	}
}

#ifdef OMPI_REMOTE_OFFLOADING

static int Moduleid(char *modulename)
{
	int i;
	for (i = 0; allmodules[i].modname != NULL; i++)
	{
		if (strcmp(allmodules[i].modname, modulename))
			return allmodules[i].modid;
	}

	return -1;
}

struct searchquery_ {
	char **searchmods;
	int nsearchmods;
};

/* Pairs of modulenames <-> nodenames, i.e. which nodes provides which module */
SET_TYPE_DEFINE(nodemodset, int, str, DEFAULT_HASHTABLESIZE);
SET_TYPE_IMPLEMENT(nodemodset);

set(nodemodset) nodemods;

/* Builds `nodemods` set */
static void handle_modules(rdev_config_module_t *mod, void *info)
{
	int i;
	struct searchquery_ *info_ = (struct searchquery_ *) info;

	for (i = 0; i < info_->nsearchmods; i++)
	{
		if (strcmp(mod->name, info_->searchmods[i]) == 0)
		{
			set_put(nodemods, Moduleid(info_->searchmods[i]))->value = Str(mod->node->name);
			return;
		}
	}
}


static void add_suffix_to_modulename(rdev_config_module_t *module, void *data)
{
	char *tmp = strndup(module->name, strlen(module->name));
	size_t newsize = RDEVCONF_MAXNAME + 8;
	module->name = realloc(module->name, newsize * sizeof(char));
	if (module->name == NULL)
	{
		perror("add_suffix_to_modulename():");
		exit(EXIT_FAILURE);
	}
	snprintf(module->name, newsize - 1, 
	        "%s_node%d", tmp, module->node - rdev_config.nodes);
	free(tmp);
}


void remote_modules_employ(void)
{
	int i, j;
	rdev_config_node_t *node;
	rdev_config_module_t *module;
	char *tmp;
	struct searchquery_ info;
	
	nodemods = set_new(nodemodset);

	rdev_config_initialize(IGNORE_DISABLED_MODULES);
	
	info.nsearchmods = rdev_config.nuniquemodules;
	info.searchmods = rdev_config.uniquemodnames;

	rdev_config_iterate(handle_modules, (void *) &info);

	/* Append "_nodeX" suffix to each remote module */
	rdev_config_iterate(add_suffix_to_modulename, NULL);
}


void remote_modules_finalize(void)
{
	int i;
	setelem(nodemodset) e;
	for (e = nodemods->first; e; e = e->next)
	{
		str_free(e->value);
	}
	set_free(nodemods);
	rdev_config_finalize();
}

#endif


char *modules_argfor_ompi()
{
	int i;
	str modstr;

#ifdef OMPI_REMOTE_OFFLOADING
	if (!nmodules && !rdev_config.nuniquemodules)
#else
	if (!nmodules)
#endif
		return ("");

	modstr = Strnew();  /* String for formating module names */

	for (i = 0; i < nmodules; i++)
		str_printf(modstr, "%s-usemod=%s", i ? " -" : "-", modulenames[i]);

#ifdef OMPI_REMOTE_OFFLOADING
		/* Include remote modules, as well */
		for (i = 0; i < rdev_config.nuniquemodules; i++)
		{
			if (!is_substr(str_string(modstr), rdev_config.uniquemodnames[i]))
				str_printf(modstr, " --usemod=%s", rdev_config.uniquemodnames[i]);
		}
#endif
	return ( str_string(modstr) );
}



#ifdef ENABLE_KERNEL_BUNDLING

char *bundle_get_tablename(char *fname)
{
	struct timeval timestamp;
	str sanitized_final;
	char *sanitized, *ext, *fname_copy = strdup(fname);

	if ((ext = strrchr(fname_copy, '.')) != NULL)
		*ext = 0; /* remove extension */

	sanitized = sanitize_str(fname_copy);

	gettimeofday(&timestamp, NULL);
	sanitized_final = Strnew();

	str_printf(sanitized_final, "_%s_%X%X_table", 
	                       sanitized, 
						   (unsigned)timestamp.tv_sec,
						   (unsigned)timestamp.tv_usec
	);

	return str_string(sanitized_final);
}

/* Runs ompiconf to create a kernel bundle.
 * 
 * If KERNEL_BUNDLING_MODE == BUNDLE_SOURCES (source bundling), this should be called instead of
 * kernel_makefiles. 
 * Otherwise, this should be called after kernel compilation.
 */
char *bundle_kernel_files(char *fname, char *tabname, int nkernels)
{
	int i, j, exitflag, res;
	str command = Strnew();
	char *fnamecopy = strdup(fname), *ext;
	char *outfile_, *objfile_;
	str outfile, objfile;

	if ((ext = strrchr(fnamecopy, '.')) != NULL)
		*ext = 0; /* remove extension */
	
	outfile = Str(fnamecopy);
	objfile = Str(fnamecopy);

	/* <filename>_kernels.{o,c} */
	str_printf(objfile, "_kernels.o");
	str_printf(outfile, "_kernels.c");

	/* Initial arguments */
	str_printf(command, "%s --bundle-kernels --primary-file %s --table-name %s ", 
	                    OmpiConfName, fname, tabname);

	/* For each kernel and for each available module, append necessary 
	 * arguments (--id --type --file) 
	 */
	for (i = 0; i < nkernels; i++)
	{
		str modstr = Strnew();  /* String for formating module names */

#if KERNEL_BUNDLING_MODE == BUNDLE_SOURCES
		str_printf(command, "--id %d --type KERNELTYPE_C --file %s_d%02d.c ", i, fnamecopy, i);
#endif
		for (j = 0; j < nmodules; j++)
		{
			modinfo_t *modinfo = get_module_info(modulenames[j]);
			if (!modinfo)
				continue;
#if KERNEL_BUNDLING_MODE == BUNDLE_SOURCES
			if (modinfo->sourcetype && (strcmp(modinfo->sourcetype, "KERNELTYPE_C")))
			{
				str_printf(command, "--id %d --type %s --file %s_d%02d%s ",
									i, modinfo->sourcetype, fnamecopy, i, modinfo->suffix);
				str_printf(modstr, " %s", modulenames[j]);
			}
				
#else
			if (modinfo->outtype)
			{
				str_printf(command, "--id %d --type %s --file %s_d%02d%s ",
									i, modinfo->outtype, fnamecopy, i, modinfo->outsuffix);
				str_printf(modstr, " %s", modulenames[j]);
			}
#endif
		}

		/* Include remote modules, as well */
		for (j = 0; j < rdev_config.nuniquemodules; j++)
		{
			if (!is_substr(str_string(modstr), rdev_config.uniquemodnames[j]))
			{
				modinfo_t *modinfo = get_module_info(rdev_config.uniquemodnames[j]);
				if (!modinfo)
					continue;
#if KERNEL_BUNDLING_MODE == BUNDLE_SOURCES
				if (modinfo->sourcetype && (strcmp(modinfo->sourcetype, "KERNELTYPE_C")))
					str_printf(command, "--id %d --type %s --file %s_d%02d%s ",
									i, modinfo->sourcetype, fnamecopy, i, modinfo->suffix);
#else
				if (modinfo->outtype)
					str_printf(command, "--id %d --type %s --file %s_d%02d%s ",
										i, modinfo->outtype, fnamecopy, i, modinfo->outsuffix);
#endif
			}
		}

		str_free(modstr);
	}
	
	/* Redirect the output to outfile */
	str_printf(command, "> \"%s\"", str_string(outfile));

	if (verbose)
		fprintf(stderr, "  [ %s ]\n", str_string(command));

	/* Finally execute the command */
	if ((res = sysexec(str_string(command), NULL)) != 0)
		_exit(res);

	str_truncate(command);
	str_printf(command, "%s \"%s\" -c", COMPILER, str_string(outfile));
	if (verbose)
		fprintf(stderr, "====> Compiling kernel bundle\n  [ %s ]\n",
						str_string(command));
	res = sysexec(str_string(command), NULL);
	if (!keep)
		unlink(str_string(outfile));
	if (res)
		_exit(res);

	str_free(command);
	str_free(outfile);
	free(fnamecopy);
	
	objfile_ = strdup(str_string(objfile));

	str_free(objfile);
	return objfile_;
}

#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        CREATING & EXECUTING MAKEFILES FOR KERNELS         *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static
void km_substitute(char *var, int maxlen, str s, char *modulename, int kernid)
{
	// FIXME: is goutfile set here?
	if (strncmp("@@OMPI_OUTPUT@@", var, maxlen) == 0)
		str_printf(s, "%s", (user_outfile.head && strlen(user_outfile.head->val) != 0) ?
		                    user_outfile.head->val : "a.out");
	else if (strncmp("@@OMPI_LIBDIR@@", var, maxlen) == 0)
		str_printf(s, LibDir);
	else if (strncmp("@@OMPI_COMMLIBARG@@", var, maxlen) == 0)
#ifdef OMPI_REMOTE_OFFLOADING
		str_printf(s, "-L%s/default_comm", LibDir);
#else
		str_printf(s, "", LibDir);
#endif
	else if (strncmp("@@OMPI_MODULE@@", var, maxlen) == 0)
		str_printf(s, "%s", modulename);
	else if (strncmp("@@OMPI_CC@@", var, maxlen) == 0)
		str_printf(s, "%s", COMPILER);
	else if (strncmp("@@OMPI_CFLAGS@@", var, maxlen) == 0)
		str_printf(s, "%s", CFLAGS);
	else if (strncmp("@@OMPI_LDFLAGS@@", var, maxlen) == 0)
		str_printf(s, "%s", LDFLAGS);
	else if (strncmp("@@OMPI_KERNELID@@", var, maxlen) == 0)
		str_printf(s, "%02d", kernid);
	else if (strncmp("@@OMPI_KERNELFILE@@", var, maxlen) == 0)
		str_printf(s, "%s", current_file);
	else if (strncmp("@@CUDA_OUTPUT_EXT@@", var, maxlen) == 0)
		str_printf(s, "%s", CUDA_KERNEL_EXTENSION);
	else
		str_printf(s, "%*.*s", maxlen, maxlen, var);
}


static
void km_substitute_vars(char *line, str s, char *modulename, int kernid)
{
	char *p;

	for (; isspace(*line); line++)  /* skip spaces */
		if (*line == 0)
			return;
		else
			str_putc(s, *line);

	if (*line == '#')
	{
		str_printf(s, "%s", line);    /* comment line */
		return;
	}

	while (1)
	{
		for (; *line != '@'; line++)
			if (*line == 0)
				return;
			else
				str_putc(s, *line);

		p = line+1;
		if (*p != '@')
			str_putc(s, '@');
		else
		{
			for (p++; *p != '@'; p++)
				if (*p == 0)
				{
					str_printf(s, "%s", line);
					return;
				}
				else
					if (isspace(*p))
					{
						str_printf(s, "%*.*s", p-line, p-line, line);
						break;
					};

			if (*p == '@')
			{
				if (*(p++) == '@')
				{
					km_substitute(line, p-line+1, s, modulename, kernid);
					p++;
				}
			}
		}
		line = p;
	}
}


static
int kernel_makefile_run(char *path, char *modulename, int kernid)
{
	str cmd = Strnew();
	int  res, eflag;
	char *currfile = strdup(current_file), *ext, *bname;

	str sshprefix = Str("");

#if defined(ENABLE_KERNEL_BUNDLING) && (KERNEL_BUNDLING_MODE == BUNDLE_BINARIES)
	/* If bundling is enabled, connect to the node that provides our module
	 * and run the kernel makefile there. */
	setelem(nodemodset) e;
	
	if ((e = set_get(nodemods, Moduleid(modulename))))
		if ((!is_module_supported(modulename)))
			str_printf(sshprefix, "ssh %s 'cd %s && ", str_string(e->value), cwd);
#endif

	if (strcmp(path, ".") != 0)
	{
		if (*cwd == 0)
		{
			perror("getcwd():");
			return (1);
		}
		if (verbose)
			fprintf(stderr, "  ===> changing to directory %s\n", path);
		if (chdir(path) < 0)
		{
			perror("chdir():");
			return (1);
		}
	}

	ext = strrchr(currfile, '.');
	*ext = 0;

	bname = basename(currfile);

	str_printf(cmd, "%smake -f %s-makefile-%s-%02d", str_string(sshprefix), modulename, bname, kernid);

	if (str_tell(sshprefix))
		str_printf(cmd, "'");

	if (verbose)
		fprintf(stderr, "  ===> %s\n", str_string(cmd));
	res = sysexec(str_string(cmd), &eflag);

	if (strcmp(path, ".") != 0)
	{
		if (verbose)
			fprintf(stderr, "  ===> returning to directory %s\n", cwd);
		if (chdir(cwd) < 0)
		{
			perror("chdir():");
			return (1);
		}
	}

	str_free(cmd);
	
	return (eflag || res);
}


int kernel_makefile_create(char *path, char *modulename, int kernid, 
                           mapmod_t mmod)
{
	static str s = NULL;
	char   *flavor, *ext;
	FILE   *fpin, *fpout;
	char   *currfile = strdup(current_file), *bname, *ext_new;
	str    line, filepath = Strnew();

	if ((ext = strrchr(currfile, '.')) != NULL)
		*ext = 0; /* remove extension */
	else
		ext = currfile + strlen(currfile);

	ext_new = strdup(currfile);

	if (!usecarstats || !mmod)
		str_printf(filepath, "%s/devices/%s/MakeKernel.%s", LibDir, modulename, modulename);
	else
	{
		snprintf(ext, PATHSIZE-strlen(currfile), "_d%02d.c", kernid);

		/* Mapper arguments */
		str_printf(filepath, "%s/%s", cwd, currfile);
		flavor = mapper_select_flavor(mmod, filepath);
		str_truncate(filepath);
		
		/* Check for actual flavor and form the kernel makefile installation path */
		if (flavor == NULL || strcmp(flavor, "devpart") == 0)
			str_printf(filepath, "%s/devices/%s/MakeKernel.%s", 
					LibDir, modulename, modulename);
		else
		{
			char *flv = flavor;    /* skip "devpart" from returned flavor name */
			if (strlen(flavor) >= 7)
				flv = (flavor[7] == 0) ? flavor+7 : flavor+8;
			str_printf(filepath, "%s/devices/%s/MakeKernel-%s.%s", 
					LibDir, modulename, flv, modulename);
		}
		if (verbose)
			fprintf(stderr, "  kernel makefile selected (mapper):\n\t%s\n", str_string(filepath));
	}

	if ((fpin = fopen(str_string(filepath), "r")) == NULL)
	{
		fprintf(stderr, "[***] cannot find 'MakeKernel.%s' recipe for creating "
		                "%s kernels.\n", modulename, modulename);
		str_free(filepath);
		return (1);
	}

	/* filename without ext */
	bname = basename(ext_new);

	str_truncate(filepath);
	str_printf(filepath, "%s/%s-makefile-%s-%02d", 
			path, modulename, bname, kernid);

	if ((fpout = fopen(str_string(filepath), "w")) == NULL)
	{
		fprintf(stderr, "[***] cannot generate '%s/%s-makefile-%s-%02d' to create "
		                "kernels for device %s\n.", path, modulename, bname,
						kernid, modulename);
		fclose(fpin);
		str_free(filepath);
		return (1);
	}

	if (s)
		str_truncate(s);
	else
		s = Strnew();

	line = Strnew();
	str_reserve(line, SLEN);
	while (fgets(str_string(line), SLEN, fpin) != NULL)
	{
		km_substitute_vars(str_string(line), s, modulename, kernid);
	}
	fprintf(fpout, "%s", str_string(s));

	fclose(fpin);
	fclose(fpout);

	if (verbose)
		fprintf(stderr, "  ( %s )\n", str_string(filepath));
	kernel_makefile_run(path, modulename, kernid);

	if (keep < 2)    /* Remove makefile */
		unlink(str_string(filepath));
	
	str_free(filepath);
	str_free(line);
	
	return (0);
}



/* Generate makefiles for each requested module @ the given path */
void kernel_makefiles(char *fname, int nkernels)
{
	int      i, j, childst = 0, njobs = 0;
	char     filepath[PATHSIZE], filepath_CUDA[PATHSIZE], filepath_OPENCL[PATHSIZE],
		     *s, *cu, *cl;
	mapmod_t mmod;

	if ((s = strrchr(fname, '/')) == NULL)
		strcpy(filepath, ".");
	else
	{
		strncpy(filepath, fname, s-fname); /* Strip filename from the path */
		filepath[s-fname] = 0;
	}

	if ((cu = strrchr(fname, '/')) == NULL)
		strcpy(filepath_CUDA, ".");
	else
	{
		strncpy(filepath_CUDA, fname, cu-fname);
		filepath[cu-fname] = 0;
	}
	
	if ((cl = strrchr(fname, '/')) == NULL)
		strcpy(filepath_OPENCL, ".");
	else
	{
		strncpy(filepath_OPENCL, fname, cl-fname);
		filepath[cl-fname] = 0;
	}
	
	if (!getcwd(cwd, PATHSIZE))
		*cwd = 0;
	
	/* Kernel makefile jobs
	 *
	 * For each module, we consider each kernel makefile execution
	 * a job, handled by a separate process. When the # of jobs > 1, 
	 * the overall kernel compilation overheads are reduced,
	 * as each job is executed concurrently.
	 * 
	 * The compilation load is statically distributed across processes,
	 * according to the requested number of jobs and the number of
	 * kernels to be compiled.
	 * 
	 * This feature can be leveraged with device options (see ompicc --devopt).
	 * It is particularly useful when dealing with CUDA kernels.
	 */

	sprintf(current_file, "%s\n", fname);
	for (i = 0; i < nmodules; i++)
	{
		mmod = usecarstats ? mapper_load_module(modulenames[i]) : NULL;

		JOB_START(kmakefiles, reqjobs, nkernels)
		{
			JOB_LOOP(kmakefiles, c)
				kernel_makefile_create(filepath, modulenames[i], c, mmod);
		}
		JOB_FINISH(kmakefiles)
		
		if (mmod)
			mapper_free_module(mmod);
	}

	/* Create kernel makefiles for all remote nodes, as well */
#if defined(ENABLE_KERNEL_BUNDLING) && (KERNEL_BUNDLING_MODE == BUNDLE_BINARIES)
	for (i = 0; i < rdev_config.nuniquemodules; i++)
	{
		if (is_module_supported(rdev_config.uniquemodnames[i])) 
			continue;
		JOB_START(kmakefiles, reqjobs, nkernels)
		{
			JOB_LOOP(kmakefiles, c)
				kernel_makefile_create(filepath, rdev_config.uniquemodnames[i], c, NULL);
		}
		JOB_FINISH(kmakefiles)
	}
#endif

	if (!keep)       /* TODO: Remove kernels!! */
	{
		strcpy(filepath, fname);
		strcpy(filepath_CUDA, fname);
		strcpy(filepath_OPENCL, fname);
		if ((s = strrchr(filepath, '.')) != NULL)
			*s = 0; /* remove extension */
		else
			s = filepath + strlen(filepath);

#ifdef HAVE_CUDA
		if ((cu = strrchr(filepath_CUDA, '.')) != NULL)
			*cu = 0; /* remove extension */
		else
			cu = filepath_CUDA + strlen(filepath_CUDA);
#endif
#ifdef HAVE_OPENCL
		if ((cl = strrchr(filepath_OPENCL, '.')) != NULL)
			*cl = 0; /* remove extension */
		else
			cl = filepath_OPENCL + strlen(filepath_OPENCL);
#endif
		for (i = 0; i < nkernels; i++)
		{
			snprintf(s, PATHSIZE-strlen(filepath), "_d%02d.c", i);
			unlink(filepath);
#ifdef HAVE_CUDA
			snprintf(cu, PATHSIZE-strlen(filepath_CUDA), "_d%02d-cuda.c", i);
			unlink(filepath_CUDA);
#endif

#ifdef HAVE_OPENCL
			snprintf(cl, PATHSIZE-strlen(filepath_OPENCL), "_d%02d-opencl.c", i);
			unlink(filepath_OPENCL);
#endif
		}
	}
}


#ifdef ENABLE_KERNEL_BUNDLING

/* Generate a single makefile for each requested module @ the given path */
void kernel_makefile_single(char *fname)
{
	int      i, j, childst = 0, njobs = 0;
	char     filepath[PATHSIZE], filepath_CUDA[PATHSIZE], filepath_OPENCL[PATHSIZE],
		     *s;
	mapmod_t mmod;

	if ((s = strrchr(fname, '/')) == NULL)
		strcpy(filepath, ".");
	else
	{
		strncpy(filepath, fname, s-fname); /* Strip filename from the path */
		filepath[s-fname] = 0;
	}

	if (!getcwd(cwd, PATHSIZE))
		*cwd = 0;

	sprintf(current_file, "%s\n", fname);

	JOB_START(kmfs, reqjobs, nmodules)
	{
		JOB_LOOP(kmfs, i)
		{
			mmod = usecarstats ? mapper_load_module(modulenames[i]) : NULL;
			kernel_makefile_create(filepath, modulenames[i], _kernid, NULL);
		
			if (mmod)
				mapper_free_module(mmod);
		}
	}
	JOB_FINISH(kmfs)
}

#endif