/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* assorted.c -- various utilities */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>
#include "str.h"
#include "assorted.h"


/* 
 * JOBS
 */

/* Returns a specific chunk of the work to be done, to the calling process */
int job_get_chunk(jobcb_t *job)
{
	int chunksize, numsiblings = job->nprocs;
	int myid = job->myid;

	if (numsiblings == 1) /* I am the only one */
	{
		job->fi = 0;
		job->li = job->worksize;
		return (job->fi != job->li);
	}

	chunksize = job->worksize / numsiblings;
	job->worksize = job->worksize % numsiblings;
	if (job->worksize) chunksize++;

	if (myid < job->worksize || job->worksize == 0)
	{
		job->fi = myid * chunksize;
		job->li = job->fi + chunksize;
	}
	else
	{
		job->fi = job->worksize * chunksize + (myid - job->worksize) * (chunksize - 1);
		job->li = job->fi + (chunksize - 1);
	}

	return (job->fi != job->li);
}


/* Creates a job of `worksize' iterations, named `jobname', that will be executed by 
 * `nprocs' processes.
 */
jobcb_t *job_new(int nprocs, int worksize)
{
	jobcb_t *job;
	int actual_nprocs;

	if (worksize == 0)
		return NULL;
		
	/* Correct the actual # of jobs */
	if ((worksize < nprocs) || (nprocs == 0))
		actual_nprocs = worksize;
	else if (nprocs < 0) /* Requested procs are not configured */
		actual_nprocs = 1;
	else
		actual_nprocs = nprocs;
	
	job = malloc(sizeof(jobcb_t));
	job->status = JOB_STOPPED;
	job->myid = 0;
	job->pid = -1;
	job->nprocs = actual_nprocs;
	job->worksize = worksize;
	job->fi = job->li = 0;

	return job;
}


/* Executes a specific job
 */
void job_run(jobcb_t *job)
{
	int j;
	pid_t newpid = (pid_t) -1;

	if (job->status != JOB_STOPPED) 
		return;

	job->status = JOB_RUNNING;

	for (j = 1; j < job->nprocs; j++)
	{
		if ((newpid = fork()) == 0)
		{
			job->myid = j;
			job->pid = newpid;
			break;
		}
	}
}


/* Kills a running job
 */
void job_kill(jobcb_t *job)
{
	int childst = 0;

	if (job->status != JOB_RUNNING) 
		return;

	/* All processes exit, except for the parent */
	if (job->pid == 0)
		exit(0);
	
	/* The parent waits */
	while ((job->nprocs > 1) && (wait(&childst) > 0));

	job->status = JOB_KILLED;
	free(job);
}


/* Creates an string declaration, initialized at the buffer's contents
 * Note: currently unused 
 */
char *create_string_decl(char *buffer, char *array_name, unsigned long nbytes)
{
	str s = Strnew();
	int i;
	char *text = NULL;
	char *temp = malloc(nbytes);
	if (!temp)
		return NULL;
	
	str_printf(s, "char *%s = \"%s\";\n", array_name, buffer);
	str_printf(s, "unsigned long %s_size = %lu;\n", array_name, nbytes);
	  
	text = strdup(str_string(s));
	str_free(s);
	
	return text;
}


/* 
 * ENCODING/DECODING
 */

/* Encodes a buffer to a hex array and dumps its C declaration */
char *encode_to_hex_array_withdecl(unsigned char *buffer, char *array_name, unsigned long nbytes,
                          bool static_decl, bool print_size)
{
	str s = Strnew();
	int i;
	char *hex = NULL;
	
	/* [static] unsigned char <filename>[<size>] = { */
	str_printf(s, "%sunsigned char %s[%lu] = {", 
	              static_decl ? "static ": "", array_name, nbytes);

	/* <byte1>, <byte2>, ... */
	for (i = 0; i < nbytes; i++) 
	{
		str_printf(s, "0x%02X", buffer[i]);
		if (i < nbytes - 1)
			str_printf(s, ",");
		else    
			str_printf(s, "\n");
	}

	str_printf(s, "};\n"); /* }; */

	if (print_size)
	{
		/* unsigned long <filename>_size = <size>; */
		str_printf(s, "unsigned long %s_size = %lu;\n", array_name, nbytes);
	}
	  
	hex = strdup(str_string(s));
	str_free(s);
	
	return hex;
}


/* Encodes a buffer to a hex array and dumps its C declaration */
char *encode_to_hex_array(unsigned char *buffer, char *array_name, unsigned long nbytes)
{
	str s = Strnew();
	int i;
	char *hex = NULL;
	
	/* { */
	str_printf(s, "{");

	/* <byte1>, <byte2>, ... */
	for (i = 0; i < nbytes; i++) 
	{
		str_printf(s, "0x%02X", buffer[i]);
		if (i < nbytes - 1)
			str_printf(s, ",");
		else    
			str_printf(s, "\n");
	}

	str_printf(s, "}"); /* } */
	  
	hex = strdup(str_string(s));
	str_free(s);
	
	return hex;
}


int is_substr(char *str, char *substr) 
{
	size_t start, end;
    char *ptr; 
	int noletterbefore, noletterafter;

	if ((str == NULL) || (substr == NULL))
		return 0;

	ptr = strstr(str, substr);
    if (ptr == NULL) 
        return 0;
    
    start = ptr - str;
    end = start + strlen(substr);
    
	noletterbefore = !isalnum(*(ptr-1));
	noletterafter = !isalnum(*(ptr+strlen(substr)));

    return ((start == 0 || noletterbefore) && (end == strlen(str) || noletterafter));
}


/* Removes all special symbols from a string, keeps only alphas and digits */
char *sanitize_str(char *str) 
{
	int i;
	char *buf = strdup(str);

	for (i = 0; buf[i] != '\0'; i++) 
		if (!isalpha(buf[i]) && !isdigit(buf[i])) 
			buf[i] = '_';

	return buf;
}


static bool is_substring_at_position(const char *str, const char *sub, int pos) 
{
    int i = 0;
    while (sub[i] != '\0') 
	{
        if (str[pos + i] != sub[i])
            return false;
        i++;
    }
    return true;
}


/* Checks if a word is contained in a string */
bool contains_word(const char *str, const char *word) 
{
	int i, word_len, str_len;
    if (str == NULL || word == NULL || strlen(word) == 0)
        return false;

    word_len = strlen(word);
    str_len = strlen(str);

    for (i = 0; i <= str_len - word_len; i++) 
        if (is_substring_at_position(str, word, i)) 
		{
            bool is_start_word = (i == 0) || !isalpha(str[i - 1]);
            bool is_end_word = (i + word_len == str_len) || !isalpha(str[i + word_len]);
            if (is_start_word && is_end_word)
                return true;
        }

    return false;
}