/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* rt_common.c -- common functionalities between the host and the devices */

#include "rt_common.h"


/* PLACES
 * Array of pointers; each one points to a place (i.e. an array of ints).
 * places[0][0] gives the total number of places;
 * places[i][0] (i > 0) gives the length of the i-th place array.
 */


int  **places_dup(int **from)
{
	int i;
	int num_of_places = numplaces(from);
	int **c_places;


	if(num_of_places <=0)
		return NULL;

	c_places = malloc(num_of_places*sizeof(int **));

	c_places[0] = malloc(sizeof(int));
	c_places[0][0] = from[0][0];

	for(i=1; i<=num_of_places; i++)
	{
		c_places[i] = malloc((placelen(from,i-1)+1)*sizeof(int));
		memcpy(c_places[i], from[i], (placelen(from,i-1)+1)*sizeof(int));
	}

	return c_places;
}


void  places_free(int **place)
{
	int n = numplaces(place) - 1;

	if(place == NULL)
		return;
	for (; n >=0; n--)
		if (placelen(place, n))
			free(place[n + 1]);
	free(place);
}


/* THREAD/TASK/KERNEL (TTK) Functions
 *
 * When using a cluster (MPI enabled) all nodes get a copy of the user's 
 * executable which contains all thread/task/kernel (TTK) functions. However, 
 * because of potential differences in architecture or memory layouts, 
 * the memory address at which each kernel function resides might be 
 * different among nodes. To solve this problem, each node creates a table 
 * that stores each TTK function's name and pointer; it is guaranteed that 
 * all TTK functions are stored in the same order, so that the **index** in 
 * the table is the same among all nodes. Consequently, nodes need only 
 * know the index in order to call the correct function; thus transmitters
 * communicate only the table indices, while receivers use the received
 * indices to get the function pointer from their own table.
 */


/**
 * @brief Dynamically adds a new function to a TTK function table
 * @param t    the function table
 * @param func pointer to the new function
 * @param name the function name (not dup'ed; can be NULL)
 */
void ttkfunctable_add(ttkfunctable_t *t, ttkfunc_t func, char *name)
{
	if (t->size >= t->capacity)
	{
		t->capacity += 10;       /* Increase (arbitrarily) by 10 entries */
		t->table = realloc(t->table, t->capacity * sizeof(ttkfuncdata_t));
		if (!t->table)
		{
			fprintf(stderr, "thread/task/kernel functable realloc failure.\n");
			exit(1);
		}
	}
	t->table[t->size].func = func;
	t->table[t->size++].name = name;   /* Not dupping anything */
}


/**
 * @brief Finds a TTK function by its function pointer
 * 
 * @param t    the TTK function table
 * @param func the function pointer
 * @return     the index in the table
 */
int ttkfunctable_findbyptr(ttkfunctable_t *t, ttkfunc_t func)
{
	int i;

	for (i = 0; i < t->size; ++i)
		if (t->table[i].func == func)
			return i;
	return -1;
}


/**
 * @brief Finds a TTK function by its function name
 * 
 * @param t    the TTK function table
 * @param func the function name
 * @return     the index in the table
 */
int ttkfunctable_findbyname(ttkfunctable_t *t, char *name)
{
	int i;

	for (i = 0; i < t->size; ++i)
		if (strncmp(t->table[i].name, name, strlen(name)) == 0)
			return i;
	return -1;
}


/**
 * @brief Returns a TTK function pointer given its table index
 * 
 * @param t    the TTK function table
 * @param func the index in the table
 * @return     the function pointer
 */
ttkfunc_t ttkfunctable_getptr(ttkfunctable_t *t, int index)
{
	if (index < 0 || index >= t->size)
		return NULL;
	else
		return t->table[index].func;
}




static ttkfunctable_t allkernfuncs;


void ort_kernfunc_register(char *name, ttkfunc_t func)
{
	ttkfunctable_add(&allkernfuncs, func, name);
}

int ort_kernfunc_findbyname(char *name)
{
	return ttkfunctable_findbyname(&allkernfuncs, name);
}

ttkfunc_t ort_kernfunc_getptr(int fid)
{
	return ttkfunctable_getptr(&allkernfuncs, fid);
}

void ort_kernfunc_cleanup(void)
{
	if (allkernfuncs.table)
	{
		free(allkernfuncs.table);
		allkernfuncs.table = NULL;
	}
}
