/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* slave.c -- All slave (remote; with rank != 0) MPI processes execute
 *            these functions. Their purpose is to respond to their
 *            master's requests. Note that the master and the slaves do
 *            **NOT** run in a shared memory environment. Communication is
 *            achieved with MPI.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <mpi.h>
#include "../../../rt_common.h"
#include "slave.h"
#include "memory.h"


/* Uncomment the following to activate debugging */
//#define DEBUG 
#ifdef DEBUG
	#define dbg(s) fprintf s;
#else
	#define dbg(s) 
#endif
#undef DEBUG

static MPI_Comm communicator; /* the MPI communicator to talk to the master */
static int cmd_params[4];     /* parameters the host sents with each call */


/* Jump table */
void cmdinit(), cmdget(), cmdput(), cmdalloc(), cmdfree(),
     cmdexecute(), cmdshutdown();
void (*cmdfunc[])() = {
	cmdinit, cmdget, cmdput, cmdalloc, cmdfree, cmdexecute, cmdshutdown
};


int read_command(void)
{
	MPI_Recv(cmd_params, 4, MPI_INT, 0, MPI_ANY_TAG, communicator, 
	         MPI_STATUS_IGNORE);
	return cmd_params[0]; /* Type of command to execute */
}


void wait_and_execute(MPI_Comm merged_comm)
{
	int cmd;

	communicator = merged_comm;

	/* loop waiting for commands */
	for (cmd=read_command(); cmd >= 0 && cmd <= MPI_CMD_LAST; cmd=read_command())
		(cmdfunc[cmd])();
	dbg((stderr, "  [mpinode slave (%d)]: illegal command %d received\n", 
	              getpid(), cmd));
	exit(2);
}


/* Made weak so that if found at load time they get overridden.
 * We need (weak) definitions to cover the case where the module is only 
 * loaded for queries e.g. by ompicc --devinfo; in such cases ORT is not 
 * present and those symbols would otherwise be undefined.
 */
#pragma weak actual_devpart_med2dev_addr
char *(*actual_devpart_med2dev_addr)(void *, unsigned long);
#pragma weak actual_omp_is_initial_device
int (*actual_omp_is_initial_device)(void);


char *mpinode_devpart_med2dev_addr(void *medaddr, unsigned long size)
{
	dbg((stderr, "  [mpinode slave %d] devpart_med2dev_addr %p --> %p\n", 
	             getpid(), medaddr, alloc_items_get((size_t)medaddr)));

	return alloc_items_get((size_t)medaddr);
}


void override_devpart_med2dev_addr(void)
{
	extern char *(*actual_devpart_med2dev_addr)(void *, unsigned long);
	actual_devpart_med2dev_addr = mpinode_devpart_med2dev_addr;
}


int mpinode_omp_is_initial_device(void)
{
	return 0;
}


void override_omp_is_initial_device(void)
{
	extern int (*actual_omp_is_initial_device)(void);
	actual_omp_is_initial_device = mpinode_omp_is_initial_device;
}


/* Host performs INIT (devid) */
void cmdinit()
{
	int devid = cmd_params[1];

	alloc_items_init(1); /* Slave process cares only for itself */
	alloc_items_init_global_vars(devid, 0);
}


/* Host performs GET (medaddress, offset, size) */
void cmdget()
{
	size_t maddr = cmd_params[1];
	int offset = cmd_params[2], nbytes = cmd_params[3];
	void *devaddr; /* The actual memory space on device */

	dbg((stderr, "  [mpinode slave %d] GET cmd from %d (offset:%d, size:%d)\n", 
	             getpid(),  maddr, offset, nbytes));

	devaddr = alloc_items_get(maddr);
	MPI_Send(devaddr + offset, nbytes, MPI_BYTE, 0, 0, communicator);
}


/* Host performs WRITE (medaddress, offset, size) */
void cmdput()
{
	size_t maddr = cmd_params[1];
	int offset = cmd_params[2], nbytes = cmd_params[3];
	void *devaddr; /* The actual memory space on device */

	dbg((stderr, "  [mpinode slave %d] PUT cmd to %p (offset:%d, size:%d)\n", 
	             getpid(), maddr, offset, nbytes));

	devaddr = alloc_items_get(maddr);
	MPI_Recv(devaddr + offset, nbytes, MPI_BYTE, 0, MPI_ANY_TAG, communicator,
			MPI_STATUS_IGNORE);
}


/* Host performs DEVALLOC (size, medaddress) */
void cmdalloc()
{
	int nbytes = cmd_params[1];
	size_t maddr = cmd_params[2];
	alloc_items_add(maddr, nbytes);

	dbg((stderr, "  [mpinode slave %d] ALLOC cmd for %d bytes --> maddr: %p\n", 
	             getpid(), nbytes, maddr));
}


/* Host performs DEVFREE (medaddress) */
void cmdfree()
{
	size_t maddr = cmd_params[1];
	alloc_items_remove(maddr);

	dbg((stderr, "  [mpinode slave %d] FREE cmd for %ld at %p\n", getpid(), 
	             maddr, alloc_items_get(maddr)));
}


/* Host performs OFFLOAD */
void cmdexecute()
{
	void *devdata; /* usuable (on host) mediary addresses of struct */
	int exec_result, kernel_id = cmd_params[1];
	size_t size = cmd_params[2], devdata_maddr = cmd_params[3];
	ttkfunc_t kernel_function;

	/* Get the kernel info */
	if (size == 0)
		devdata = NULL;
	else
	{
		devdata = alloc_items_add(devdata_maddr, size);
		MPI_Recv(devdata, size, MPI_BYTE, 0, MPI_ANY_TAG, communicator,
		         MPI_STATUS_IGNORE); /* Memory copy */

		dbg((stderr, "  [mpinode slave %d] EXECUTE cmd: alloced devdata "
		        "(i:%d, u:%p, size:%d)\n", getpid(), devdata_maddr, devdata, size));
	}

	kernel_function = ort_kernfunc_getptr(kernel_id);

	dbg((stderr, "  [mpinode slave %d] EXECUTE cmd: start kernel (id=%d @ %p)\n",
			         getpid(), kernel_id, kernel_function));
	
	kernel_function((void *)devdata); /* call the kernel */

	if (devdata)
		alloc_items_remove(devdata_maddr);

	exec_result = 11; /* all OK */
	MPI_Send(&exec_result, 1, MPI_INT, 0, 0, communicator);   /* Notify host */

	dbg((stderr, "  [mpinode slave %d] EXECUTE cmd: finish kernel (id %d)\n", 
	             getpid(), kernel_id));
}


/* Host performs SHUTDOWN */
void cmdshutdown()
{
	dbg((stderr, "  [mpinode slave %d] >>> SHUTDOWN cmd; exitting...\n",
	             getpid()));

	alloc_items_free_all();
	ort_kernfunc_cleanup();
	MPI_Comm_free(&communicator);
	MPI_Finalize();
	exit(0);
}
