/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <assert.h>
#include "x_cancel.h"
#include "ast_xform.h"
#include "ast_free.h"
#include "ast_copy.h"
#include "ast_assorted.h"
#include "ast_arith.h"
#include "ast.h"
#include "ompi.h"

#include "x_clauses.h"

static void cancel_error(aststmt *t, char *ctype, char *etype)
{
	exit_error(1, "(%s, line %d) openmp error:\n\t"
	           "\"cancel %s\" must be closely nested inside a \"%s\" construct\n",
	           (*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l,
	           ctype, etype);
}


void xform_cancel_type(aststmt *t, int *type, char label[22])
{
	ompclause c;
	ompcon    enclosing_con = ast_get_enclosing_ompcon((*t)->parent, 0);

	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCPARALLEL)) != NULL)
	{
		if (enclosing_con->type != DCPARALLEL)
			cancel_error(t, "parallel", "parallel");

		*type = 0;
	}
	else
		if ((c = xc_ompcon_get_clause((*t)->u.omp, OCTASKGROUP)) != NULL)
		{
			if (enclosing_con->type != DCTASK)
				cancel_error(t, "taskgroup", "task");

			*type = 1;
		}
		else
			if ((c = xc_ompcon_get_clause((*t)->u.omp, OCFOR)) != NULL)
			{
				if (enclosing_con->type != DCFOR && enclosing_con->type != DCFOR_P)
					cancel_error(t, "for", "for");

				*type = 2;
			}
			else
			{
				c = xc_ompcon_get_clause((*t)->u.omp, OCSECTIONS);
				if (enclosing_con->type != DCSECTIONS)
					cancel_error(t, "sections", "sections");

				*type = 3;
			}
	assert(c != NULL);

	snprintf(label, 22, "CANCEL_%s_%d", ompdirnames[enclosing_con->type],
	         enclosing_con->l);
}


void xform_cancel(aststmt *t)
{
	astexpr   ifexpr = NULL;
	aststmt   parent = (*t)->parent, v;
	int       type = -1;
	char      label[22];
	ompclause c;

	/* Find and store "if" clause */
	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCIF)) != NULL)
		ifexpr = ast_expr_copy(c->u.expr);

	/* Get label and type */
	xform_cancel_type(t, &type, label);

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);

	if (ifexpr == NULL)
		/* if (ort_enable_cancel(<type>))
		 *   goto <label>;
		 */
		*t = BlockList(v,
		               If(
		                 FunctionCall(
		                   IdentName("ort_enable_cancel"), numConstant(type)
		                 ),
		                 Goto(Symbol(label)),
		                 NULL
		               )
		              );
	else
		/* if (<ifexpr>)
		 *   if (ort_enable_cancel(<type>))
		 *     goto <label>;
		 * else
		 *   if (ort_check_cancel(<type>))
		 *     goto <label>;
		 */
		*t = BlockList(v,
		               If(ifexpr,
		                  If(
		                    FunctionCall(
		                      IdentName("ort_enable_cancel"), numConstant(type)
		                    ),
		                    Goto(Symbol(label)),
		                    NULL
		                  ),
		                  If(
		                    FunctionCall(
		                      IdentName("ort_check_cancel"), numConstant(type)
		                    ),
		                    Goto(Symbol(label)),
		                    NULL
		                  )
		                 )
		              );
	ast_stmt_parent(parent, *t);
}


void xform_cancellationpoint(aststmt *t)
{
	aststmt   parent = (*t)->parent, v;
	int       type = -1;
	char      label[22];

	/* Get label and type */
	xform_cancel_type(t, &type, label);

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);

	/* if (ort_check_cancel(<type>))
	 *   goto <label>;
	 */
	*t = BlockList(v,
	               If(
	                 FunctionCall(
	                   IdentName("ort_check_cancel"), numConstant(type)
	                 ),
	                 Goto(Symbol(label)),
	                 NULL
	               )
	              );

	ast_stmt_parent(parent, *t);
}
