/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include "dev_manager.h"
#include "slave.h"
#include "../../../host/ort_prive.h"

/* Uncomment the following to activate debugging */
//#define DEBUG 
#ifdef DEBUG
	#define dbg(s) fprintf s;
#else
	#define dbg(s) 
#endif
#undef DEBUG

#define DBGTERMINAL "/usr/bin/xterm" /* xterm used for debugging */

static int *argc;
static char ***argv;
static char **node_names; /* The nodes we will use */
static int num_nodes = 0; /* # nodes */

#define CONFIGURED() (num_nodes > 0)


static char *current_node_info()
{
	static char name[MPI_MAX_PROCESSOR_NAME], string[MPI_MAX_PROCESSOR_NAME+64];
	static int name_len;

	/* It is assumed that MPI has been initialized already... */
	MPI_Get_processor_name(name, &name_len);
	sprintf(string, "node %s (pid = %d)", name, getpid());
	return string;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * CONFIGURATION FILE                                                *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static void node_add(char *dev_name, size_t len)
{
	static int node_names_capacity = 0;

	if (num_nodes == node_names_capacity)
	{
		node_names_capacity += 16;
		node_names = realloc(node_names, node_names_capacity * sizeof(char *));
		if (!node_names)
		{
			perror("add_dev_name");
			exit(EXIT_FAILURE);
		}
	}
	node_names[num_nodes++] = strndup(dev_name, len);
}


static void process_config_line(char *line)
{
	char *c, *start, *end;

	for (c = line; *c; ++c) /* Skip spaces and ignore comment */
	{
		if (*c == '#')
			return;
		else if (!(isspace(*c)))
			break;
	}
	if (!(*c))
		return;

	start = c;
	while (*c && !isspace(*c) && (*c != '#'))
		++c;
	end = c;

	node_add(start, end - start);
}


static int process_config_file()
{
	FILE *cfgfile;
	char *line = NULL, *homedir, filepath[512];
	size_t len;

	/* create full configuration file path */
	homedir = getenv("HOME");
	len = strlen(homedir);
	strncpy(filepath, homedir, 511);
	if (len < 511)
		strncpy(filepath + len, MPINODE_CONFIG, 511 - len);

	if ((cfgfile = fopen(filepath, "r")) == NULL)
		return 1;

	while (getline(&line, &len, cfgfile) != -1) /* process lines */
		process_config_line(line);

	fclose(cfgfile);
	if (line)
		free(line);
	return 0;
}


/* Produces a freeable string with the node names, spearated by commas */
static char *node_names_to_string(void)
{
	char *names;
	int i;
	size_t length = 0;

	for (i = 0; i < num_nodes; ++i) /* calculate length */
		length += strlen(node_names[i]);
	length += num_nodes; /* for the commas and the \0 */

	if ((names = calloc(length, 1)) == NULL)
	{
		perror("node_names_to_string");
		exit(1);
	}

	/* concatenate the names */
	for (length = i = 0; i < num_nodes; ++i)
	{
		strcpy(names + length, node_names[i]);
		length += strlen(node_names[i]);
		if (i < num_nodes - 1)
			names[length++] = ',';
	}
	names[length] = 0;

	return names;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * DEVICE HANDLING                                                   *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


int devman_get_num_devices()
{
	if (!CONFIGURED())
		process_config_file();
	return num_nodes;
}


char *devman_get_devicename(int devid)
{
	return node_names[devid];
}


void devman_finalize()
{
	int i;

	for (i = 0; i < num_nodes; ++i)
		free(node_names[i]);
	free(node_names);
	node_names = NULL;

	dbg((stderr, "[mpinode] closing; devman_finalize was called\n"));
}


/* Check to see if MPI has been initialized on not; if not, do it. */
static void init_mpi()
{
	int mpiinited = 0, provided;

	MPI_Initialized(&mpiinited);
	if (!mpiinited)
	{
		MPI_Init_thread(argc, argv, MPI_THREAD_MULTIPLE, &provided);
		if (provided != MPI_THREAD_MULTIPLE)
		{
			fprintf(stderr, "Your MPI library was not configured with "
			                "MPI_THREAD_MULTIPLE support. Aborting...\n");
			MPI_Abort(MPI_COMM_WORLD, 1);
		}
		dbg((stderr, "[mpinode] MPI initialized @ %s\n", current_node_info()));
	}
	else
	{
  	dbg((stderr, "[mpinode] MPI was already initialized @ %s\n", 
		     current_node_info()));
	}
}


/* This is called by the host and the other nodes upon first
 * initialization (i.e. during hm_get_num_devices())
 */
static void init_current_node(void)
{
	extern void ort_init_after_modules(void);
	extern ort_vars_t *ort; /* Pointer to ort_globals */
	MPI_Comm parent_comm, merged_comm;

	init_mpi();
	MPI_Comm_get_parent(&parent_comm); 
	if (parent_comm == MPI_COMM_NULL)     /* I am the master; done for now */
		return;

	/* All other nodes (slaves) come here; they will have to block waiting
	 * for commands from the master instead of executing user's code.
	 */

	/* Slaves set high = 1 (second arg) so the master process gets rank = 0 in
	 * the merged communicator. 
	 */
	MPI_Intercomm_merge(parent_comm, 1, &merged_comm);

	dbg((stderr, "[mpinode] I'm a SLAVE @ %s\n", current_node_info()));

	override_devpart_med2dev_addr();
	override_omp_is_initial_device();

	/* Slave processes should continue ORT initialization; now they
	 * are stuck at ort_discover_modules(). For the global variables
	 * registration to work, we also need to know the correct number
	 * of devices; host provides us with it.
	 */
	MPI_Bcast(&(ort->num_devices), 1, MPI_INT, 0, merged_comm);
	ort_init_after_modules();

	dbg((stderr, "[mpinode] I'm a SLAVE @ %s -- blocking...\n", 
	     current_node_info()));

	wait_and_execute(merged_comm);
}


/* This is called by the host and the other nodes (when spawned)
 * from hm_get_num_devices().
 */
void devman_init()
{
	dbg((stderr, "[mpinode] devman starting init (pid = %d)\n", getpid()));
	init_current_node();     /* Only the master returns from this */
	if (!CONFIGURED())       /* This is called from 2 places... */
		process_config_file();
}


/* Only the host calls this */
static MPI_Comm spawn_all_devices(void)
{
	MPI_Info info;
	MPI_Comm spawned_comm, merged_comm;
	int omp_get_num_devices(void);
	int num_total_devices;
	char *all_devs_string = node_names_to_string();

	dbg((stderr, "[mpinode] master spawning all %d \"device(s)\" {\n", 
	     devman_get_num_devices()));

	init_mpi();
	if (MPI_Info_create(&info) != MPI_SUCCESS)
	{
		fprintf(stderr, "[spawn_all_devices] MPI_Info_create failure\n");
		return MPI_COMM_NULL;
	}
	if (MPI_Info_set(info, "host", all_devs_string) != MPI_SUCCESS)
	{
		fprintf(stderr, "[spawn_all_devices] MPI_Info_set failure\n");
		return MPI_COMM_NULL;
	}

#ifdef DEBUG_NEVER
	char *tmp_argv[] = {"-e", "gdb", *argv[0], NULL};
	dbg("about to spawn %s (first arg=%p/*p=%p)\n", **argv, *argv + 1, *(*argv + 1));
	MPI_Comm_spawn(DBGTERMINAL, tmp_argv, devman_get_num_devices(), info, 0,
								 MPI_COMM_WORLD, &spawned_comm, MPI_ERRCODES_IGNORE);
#else
	if (MPI_Comm_spawn(**argv, *argv + 1, devman_get_num_devices(), info, 0,
						 MPI_COMM_WORLD, &spawned_comm, MPI_ERRCODES_IGNORE) != MPI_SUCCESS)
	{
		fprintf(stderr, "[spawn_all_devices] MPI_Comm_spawn failure\n");
		return MPI_COMM_NULL;
	}
#endif

	MPI_Info_free(&info);
	free(all_devs_string);

	/* Master sets high = 0 (second arg) so the master process gets rank = 0 in
	 * the merged communicator. 
	 */
	MPI_Intercomm_merge(spawned_comm, 0, &merged_comm);

	/* Broadcast the correct number of devices to slave nodes to finish
	 * their initialization.
	 */
	num_total_devices = omp_get_num_devices();
	MPI_Bcast(&num_total_devices, 1, MPI_INT, 0, merged_comm);

	dbg((stderr, "[mpinode] } MPI finished spawing\n"));

	return merged_comm;
}


/* This one is called only from the host, during hm_initialize().
 * It spawns the slaves and returns the merged communicator.
 */
MPI_Comm devman_create_devices(int *user_argc, char ***user_argv)
{
	dbg((stderr, "[mpinode] master creating devices (pid = %d)\n", getpid()));

	if (!CONFIGURED())
		process_config_file();
	if (num_nodes == 0)
		return MPI_COMM_NULL;

	argc = user_argc;
	argv = user_argv;
	return ( spawn_all_devices() ); /* the actual spawning is here */
}
