/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* tac.h - Three-address code and related stuff */

#ifndef __TAC_H__
#define __TAC_H__

#include <set.h>
#include "ompi.h"

/* A variable can be mentioned/accessed in the following different ways:
 * READ  - the variable is read
 * WRITE - the variable is written
 * REFER - the address of the variable is taken
 * DEREF - the variable is dereferenced (*pointer)
 * NAMED - the variable was mentioned but in none of the above ways.
 *         For example, it was the operand of a sizeof() operator.
 * CANT  - cannot decide if it was mentioned or not 
 * NONE  - the variable was not mentioned at all
 */
typedef enum { /* Do *NOT* change the values; they must match accsum_e */
	ACC_NONE=0, ACC_READ=1, ACC_WRITE=2, ACC_REFER=8, ACC_DEREF=16, 
	ACC_NAMED=32, ACC_CANT=64
} accway_e;

/* A table of all variable accesses, in program order; a variable may
 * appear multiple times if accessed more than once.
 */
typedef struct {
	symbol   var;
	accway_e way;
} acclistelem_t;

typedef struct {
	acclistelem_t *list;
	int nelems;
} acclist_t;

/* Given an expression, list all variable accesses in program order */
extern void expr2acclist(astexpr e, acclist_t *l);

/* If a variable is accessed (possibly many times), here is a summary of
 * all the ways it may have been accessed:
 *   xR - the variable was only read
 *   xW - the variable was only written
 *   xRW - the variable was first read and then written
 *   xWR - the variable was first wrtten and then read
 *   xNA - not read, not written
 * In addition, a variable may also be mentioned as follows:
 *   xREF - the variable was referenced (address taken)
 *   xDRF - the variable was dereferenced (*pointer)
 *   xNAM - the variable was named in a sizeof() operator
 */
#define ACCRW(x)    ((x) & 7)
#define ACCREF(x)   ((x) & xREF)
#define ACCDEREF(x) ((x) & xDRF)
#define ACCNAMED(x) ((x) & xNAM)
typedef enum { /* Do *NOT* change the values; they must match accway_e */
	xNA = 0, xR = 1, xW = 2, xRW = 3, xWR = 4, xREF = 8, xDRF = 16, xNAM = 32
} accsum_e;


SET_TYPE_DEFINE(accvar, symbol, accsum_e, 11)

/* Given the variable accesses list, construct a per-variable summary */
extern set(accvar) acclist2vars(acclist_t *l);

/* Debug */
extern void acclist_show(acclist_t *l);

#endif /* __TAC_H__ */
