/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*  x_clauserules.c -- accepted clauses for omp directives
 *
 * 
 * Clause rules
 * ------------
 * 
 * A clause rule describes the clauses accepted by a specific directive.
 * A ruleset is just a set of rules, grouped arbitrarily.
 * The set of all rulesets is what we keep here.
 * 
 * A rule consists of a directive type and a string which contains the 
 * allowed clauses. Within the string, clauses are separated by pipes (|).
 * 
 * A clause name is optionally followed by a list of modifiers, enclosed in
 * parentheses and separated by commas, e.g. "if (parallel, simd)"
 * 
 * In addition, a clause name is optionally followed by a symbol that 
 * denotes the maximum number of times a clause may appear. Symbols:
 *   * -- for unlimited appearances, e.g. "private*"
 *  :N -- up to N appearances allowed, e.g. "if (target, parallel):2"
 * If such symbol is not given, it defaults to :1.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "x_clauserules.h"
#include "set.h"

#define MAX_DIRCLAUSES 50
#define MAX_CLMODIFS   50
#define MAX_MERGERULES 50

static bool rules_inited = false;

#define RULESET_DEFINE(t) rule_t *t ## _dirrules = (rule_t [])
#define RULESET(t)        &t ## _dirrules
#define RULESEND          { DCNONE, "" }
typedef struct
{
	ompdirt_e dirtype;
	char *clausestr;
} rule_t;


struct {
	ompdirt_e dirtype;
	struct
	{   
		int  maxtimes;             /* Maximum # appearances allowed */
		int  nmodifs;              /* Number of modifiers supported */
		bool ocms[OCM_lastclmod];
	} clauses[OCLASTCLAUSE];
} supported_dirs[DCLASTDIR];


/************************************************
 *                                              *
 *                CLAUSE RULES                  *
 *                                              *
 ************************************************/


RULESET_DEFINE(assorted) 
{
	{ DCCANCEL, "parallel* | sections* | for* | taskgroup* | if(cancel)" },
	{ DCCANCELLATIONPOINT, "parallel* | sections* | for* | taskgroup*" },
	{ DCORDERED, "depend*" },
	{ DCCRITICAL, "hint*" },
	RULESEND
};

RULESET_DEFINE(decltarg) 
{
	{ DCDECLTARGET, "to* | link*" },
	RULESEND
};

RULESET_DEFINE(distribute)
{
	{ DCDISTSIMD, 
	"reduction* | private* | firstprivate* | lastprivate*"
	"| if(simd) | collapse | dist_schedule" },
	{ DCDISTPARFOR, 
	"reduction* | shared | copyin | private*"
	"| firstprivate* | lastprivate* | if(parallel) | num_threads"
	"| default | schedule | collapse | proc_bind"
	"| auto | dist_schedule" },
	{ DCDISTPARFORSIMD, 
	"reduction* | shared | copyin | private*"
	"| firstprivate* | lastprivate* | if(parallel, simd):2 | num_threads"
	"| default | schedule | collapse | proc_bind"
	"| dist_schedule" },
	{ DCDISTRIBUTE, 
	"private* | firstprivate* | lastprivate* | collapse"
	"| dist_schedule"
	},
	RULESEND
};

RULESET_DEFINE(for_) 
{
	{ DCFOR, "reduction* | private* | firstprivate* | lastprivate* | "
	         "ordered | orderednum | schedule | nowait | collapse | tag " },
	{ DCFOR_P, "reduction* | private* | firstprivate* | lastprivate* | "
	           "ordered | orderednum | schedule | collapse | tag " },
	{ DCFORSIMD, "reduction* | private* | firstprivate* | lastprivate* | "
	             "if(simd) | ordered | schedule | nowait | collapse | tag" },
	RULESEND
};

RULESET_DEFINE(parallel) 
{
	{ DCPARALLEL, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | if(parallel) | num_threads | default "
		"| proc_bind | auto"
	},
	{ DCPARFOR, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | lastprivate* | if(parallel) | num_threads "
		"| default | ordered | orderednum | schedule "
		"| collapse | proc_bind | auto | tag"
	},
	{ DCPARFORSIMD, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | lastprivate* | if(parallel,simd):2 | num_threads "
		"| default | ordered | schedule | collapse "
		"| proc_bind | auto"
	},
	{ DCPARSECTIONS, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | lastprivate* | if(parallel) | num_threads "
		"| default | proc_bind | auto"
	},
	RULESEND
};

RULESET_DEFINE(sections) 
{
	{ DCSECTIONS, "reduction* | private* | firstprivate* | lastprivate* | nowait" },
	RULESEND
};

RULESET_DEFINE(simd) 
{
	{ DCSIMD, "reduction* | private* | lastprivate* | if(simd) | collapse" },
	RULESEND
};

RULESET_DEFINE(single) 
{
	{ DCSINGLE, "private* | firstprivate* | copyprivate* | nowait" },
	RULESEND
};

RULESET_DEFINE(task) 
{
	{ DCTASK, "shared* | private* | firstprivate* | if(task)" 
	          "| final | mergeable | default | untied | depend*"
	          "| priority*"
	},
	RULESEND
};

RULESET_DEFINE(teams) 
{
	{ DCTEAMS, 
		"reduction* | shared* | private* | firstprivate* "
		"| default | num_teams | thread_limit"
	},
	{ DCTEAMSDISTPARFOR, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | lastprivate* | if(parallel) | num_threads "
		"| default | collapse | num_teams | thread_limit"
		"| dist_schedule | schedule"
	},
	{ DCTEAMSDISTPARFORSIMD, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | lastprivate* | if(parallel,simd):2 | num_threads "
		"| default | collapse | num_teams | thread_limit"
		"| dist_schedule | schedule"
	},
	{ DCTEAMSDIST, "reduction* | shared* | private* | firstprivate* | default "
	  "| collapse | num_teams | thread_limit | dist_schedule"
	},
	{ DCTARGETTEAMS, "reduction* | shared* | private* | firstprivate* "
		"| map* | is_device_ptr* | defaultmap* | if(target) "
		"| device | default | nowait | depend* "
		"| num_teams | thread_limit"
	},
	{ DCTARGETTEAMSDIST, "reduction* | shared* | private* | firstprivate* "
		"| map* | is_device_ptr* | defaultmap* | if(target) "
		"| device | default | nowait | collapse "
		"| depend* | num_teams | thread_limit"
		"| dist_schedule"
	},
	{ DCTARGETTEAMSDISTPARFOR, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | lastprivate* | map* | is_device_ptr* "
		"| defaultmap* | if(target, parallel):2 | num_threads | device "
		"| default | schedule | nowait | collapse "
		"| depend* | num_teams | thread_limit"
		"| dist_schedule"
	},
	{ DCTARGETTEAMSDISTPARFORSIMD, "reduction* | shared* | copyin* | private* "
		"| firstprivate* | lastprivate* | map* | is_device_ptr* "
		"| defaultmap* | if(target,parallel,simd):3 | num_threads | device "
		"| default | schedule | nowait | collapse "
		"| depend* | num_teams | thread_limit"
		"| dist_schedule"
	},
	{ DCTARGETTEAMSDISTSIMD, "reduction* | shared* | private* | firstprivate* "
		"| map* | is_device_ptr* | defaultmap* | if(target,simd):2 "
		"| device | default | nowait | collapse "
		"| depend* | num_teams | thread_limit"
		"| dist_schedule"
	},
	{ DCTEAMSDISTSIMD, "reduction* | shared* | private* | firstprivate* "
		"| default | collapse | num_teams | thread_limit "
		"| if(simd) | dist_schedule"
	},
	RULESEND
};

RULESET_DEFINE(target) 
{
	{ DCTARGET, "if (target) | private* | firstprivate* | map* | is_device_ptr*"
	            "| defaultmap* | device | nowait | depend* " },
	{ DCTARGETUPD, "from* | to* | if(target update) | device | nowait | depend*" },
	{ DCTARGETDATA, "map* | use_device_ptr* | if(target data) | device" },
	{ DCTARGETPARALLEL, "reduction* | shared* | private* | firstprivate* | map*"
	                    "| is_device_ptr* | defaultmap* | if(target,parallel):2 "
	                    "| num_threads | device | default | nowait | depend*" },
	{ DCTARGETPARFOR, "reduction* | shared* | private* | firstprivate* | lastprivate*"
	                  "| map* | is_device_ptr* | defaultmap*"
	                  "| if (target,parallel):2 | num_threads | device | default"
	                  "| ordered | orderednum | schedule | nowait | depend*"
					  "| collapse" },
	{ DCTARGENTERDATA, "map* | if(target enter data) | device | nowait | depend*" },
	{ DCTARGEXITDATA, "map* | if(target exit data) | device | nowait | depend*" },
	RULESEND
};

rule_t **allrules[] = 
{ 
	RULESET(target),   RULESET(parallel),   RULESET(sections),
	RULESET(single),   RULESET(for_),       RULESET(teams), 
	RULESET(assorted), RULESET(task),       RULESET(decltarg),
	RULESET(simd),     RULESET(distribute), NULL
};


/************************************************
 *                                              *
 *            CLAUSE RULES PARSER               *
 *                                              *
 ************************************************/


static char *trim(char *str)
{
	char *end;
	
	while (isspace((unsigned char)*str)) 
		str++;
	if (*str == 0) 
		return str;

	end = str + strlen(str) - 1;
	while (end > str && isspace((unsigned char)*end)) 
		end--;
	end[1] = 0;
	return str;
}


static bool is_number(char *s) 
{ 
	for (; *s; s++) 
		if (!isdigit(*s)) 
			return false; 
	return true; 
} 


/**
 * Search for a specific modifier in the modifier
 * names array.
 *
 * @param str The modifier string
 * @return The index of the modifier
 */
static int modifier_id(char *str)
{
	int i = 0;

	for (i = OCM_none+1; i < OCM_lastclmod; i++)
		if (strcmp(clausemods[i], str) == 0)
			return i;
	return OCM_none;
}


/* clausemodifiers: string [ `,' string ...] */
static
int parse_clrule_modifiers(ompdirt_e dirtype, char *clause_modif)
{
	char *modifiers[MAX_CLMODIFS];
	char clause[256], modtext[256], *mof = modtext;
	char *ptr = &(clause_modif[0]);
	int hasmodif = 0, nmodifs = 0, ended = 0, i, c, clausenum = 0, modifnum = 0;

	for (c = 0; *ptr != 0; ptr++)
	{
		if (*ptr == '(')
		{
			hasmodif = 1;
			break;
		}
		else 
			if (isspace(*ptr))
				continue;

		clause[c] = *ptr;
		c++;
	}
	clause[c++] = 0;

	clausenum = clr_clause_id(clause);
	assert(clausenum != OCNOCLAUSE);      /* sanity */

	if (!hasmodif)
		return clausenum;
	
	assert(*(ptr+1) != ')');
	ptr++;

	while(isspace(*ptr))
		ptr++;

	for (c = 0; *ptr != 0; ptr++)
	{
		if (*ptr == ',')
		{
			mof[c] = 0;
			c = 0;

			assert(strcmp(mof, "") != 0); /* avoid "(,..." */

			while (isspace(*mof))
				mof++;

			modifiers[nmodifs++] = strdup(trim(mof));
			
			/* Empty the string */
			memset(mof, 0, strlen(mof));

			assert(*(ptr+1) != ')'); /* avoid "(modif,)" */
			
			while(isspace(*(ptr+1)))
				ptr++;

			continue;
		}

		mof[c++] = *ptr;

		if (*(ptr+1) == ')')
		{
			mof[c++] = 0;
			modifiers[nmodifs++] = strdup(trim(mof));
			ended = 1;
			break;
		}
	}

	assert(ended);

	for (i = 0; i < nmodifs; i++)
	{
		/* Modifier must exist */
		assert((modifnum = modifier_id(modifiers[i])) != -1);
		supported_dirs[dirtype].clauses[clausenum].ocms[modifnum] = true;
		supported_dirs[dirtype].clauses[clausenum].nmodifs++;
	}

	return clausenum;
}


/* clauserule: clause[`(' clausemodifiers `)'][[`:'] multiplicity] */
/* multiplicity: `*' | integer */
static
void parse_clrule(ompdirt_e dirtype, char *str)
{
	char clause[256];
	char multiplicity[10];
	int hasmult = 0, mult = 1;
	int i, c;
	int clausenum;
	char *ptr = str;

	while (isspace(*ptr))
		ptr++;

	for (i = 0, c = 0; ptr[i] != 0; i++)
	{
		if (ptr[i] == ':')
		{
			hasmult = 1;

			assert(ptr[i+1] != 0); /* avoid "clause:" */
			break;
		}
		else if (ptr[i] == '*')
		{
			clause[c] = 0;
			mult = -1;
			break;
		}

		clause[c++] = ptr[i];
	}
	clause[c++] = 0;

	clausenum = parse_clrule_modifiers(dirtype, clause);

	/* Handle clause rule multiplicity */
	if (hasmult)
	{	
		for (i++, c = 0; ptr[i] != 0; i++)
		{
			if (isspace(ptr[i]))
				continue;
			multiplicity[c++] = ptr[i];
		}

		multiplicity[c] = 0;
		if (strcmp(multiplicity, "*") == 0)
			mult = -1;
		else 
			assert(is_number(multiplicity) && ((mult = atoi(multiplicity)) >= 1));
	}

	supported_dirs[dirtype].clauses[clausenum].maxtimes = mult;
}


/* clauserules: [ clauserule [ `|' clauserules ... ] ] */
static
void parse_directive_clrules(ompdirt_e dirtype, char *clause_rulestr) 
{
	int i, c, nkeywords = 0;
	char *rules[MAX_DIRCLAUSES];
	char newrule[256];

	for (i = 0, c = 0; clause_rulestr[i] != 0; i++)
	{
		if (clause_rulestr[i] == '|')
		{
			newrule[c] = 0;
			c = 0;
			assert(strcmp(newrule, "") != 0); /* avoid "rule||..." */
			rules[nkeywords++] = strndup(newrule, strlen(newrule));

			/* Empty the string */
			memset(newrule, 0, strlen(newrule));

			assert(clause_rulestr[i+1] != 0); /* avoid "rule1|rule2|" */
			continue;
		}

		newrule[c++] = clause_rulestr[i];

		if (clause_rulestr[i+1] == 0)
			newrule[c++] = 0;
	}

	rules[nkeywords++] = strndup(newrule, strlen(newrule));

	/* Parse each clause rule separately */
	for (i = 0; i < nkeywords; i++)
		parse_clrule(dirtype, rules[i]);
}



/************************************************
 *                                              *
 *            MAIN INTERFACE                    *
 *                                              *
 ************************************************/


/**
 * Search for a specific clause in the clause names array.
 *
 * @param str The clause string
 * @return The index of the clause
 */
ompclt_e clr_clause_id(char *str)
{
	int i;

	/* Special cases */
	if (strcmp(str, "defaultmap") == 0)
		return OCDEFAULTMAP;
	else 
		if (strcmp(str, "orderednum") == 0)
			return OCORDEREDNUM;

	for (i = OCNOCLAUSE+1; i < OCLASTCLAUSE; i++)
		if (strcmp(clausenames[i], str) == 0)
			return i;

	return OCNOCLAUSE;
}


/**
 * Search for a specific construct in the directive names array.
 *
 * @param str The construct string
 * @return The index of the construct
 */
ompdirt_e clr_dir_id(char *str)
{
	int i;

	for (i = DCNONE+1; i < DCLASTDIR; i++)
		if (strcmp(ompdirnames[i], str) == 0)
			return i;

	return DCNONE;
}


/**
 * Get allowed clause multiplicity for a directive
 *
 * @param  dirtype The directive type
 * @param  cltype  The clause type
 * @return >= 1 if clause is supported finite times, 0 if it's not
 *         supported at all, or -1 if it's supported infinite times
 */
int clr_multiplicity(ompdirt_e dirtype, ompclt_e cltype) 
{
	if (rules_inited && dirtype > DCNONE && dirtype < DCLASTDIR
	                 && cltype > OCNOCLAUSE && cltype < OCLASTCLAUSE)
		return (supported_dirs[dirtype].clauses[cltype].maxtimes);
	return 0;
}


/**
 * Check if a directive supports a specific clause
 *
 * @param  dirtype The directive type
 * @param  cltype  The clause type
 * @return 1 if given directive supports given clause, otherwise 0
 */
bool clr_allowed(ompdirt_e dirtype, ompclt_e cltype) 
{
	return (clr_multiplicity(dirtype, cltype) != 0);
}


/**
 * Check if a directive supports a specific clause with a given modifier
 *
 * @param dirtype The directive type
 * @param cltype  The clause type
 * @param modif   The modifier type
 * @return true if given directive supports given clause, otherwise 0
 */
bool clr_allowed_mod(ompdirt_e dirtype, ompclt_e cltype, ompclmod_e modif) 
{
	if (clr_multiplicity(dirtype, cltype) == 0 || modif <= OCM_none || 
	    modif >= OCM_lastclmod) 
		return false;
	return (supported_dirs[dirtype].clauses[cltype].ocms[modif]);
}


/**
 * Print all directives with their supported clauses
 */
void clr_print_all_dirs(void)
{
	int i;

	assert(rules_inited); /* rules should be already parsed */
	fprintf(stderr, ">> OMPi supported clauses for all directives:\n\n");
	for (i = 1; i < DCLASTDIR; i++)
		clr_print_dir_clauses(i);
}


/**
 * Print all clauses with the directives they're supported in
 */
void clr_print_all_clauses(void)
{
	int i;

	assert(rules_inited); /* rules should be already parsed */
	fprintf(stderr, ">> OMPi supported directives for all clauses:\n\n");
	for (i = 1; i < OCLASTCLAUSE; i++)
		clr_print_clause_dirs(i);
}


/**
 * Print all supported clauses for a specific directive
 *
 * @param dirtype The directive type
 */
void clr_print_dir_clauses(ompdirt_e dirtype)
{
	int i, j, mods = 0, clnum;

	assert(rules_inited); /* rules should be already parsed */
	assert(dirtype != 0 && dirtype < DCLASTDIR); /* dirtype must be valid */
	fprintf(stderr, "----- clauses of the '%s' directive -----\n", 
	                ompdirnames[dirtype]);

	for (i = OCNOCLAUSE+1; i < OCLASTCLAUSE; i++)
	{
		clnum = supported_dirs[dirtype].clauses[i].maxtimes;
		if (!clnum)
			continue;
		
		if (clnum == -1)
			fprintf(stderr, "            ");
		else
			fprintf(stderr, "   (max: %d) ", clnum);

		fprintf(stderr, " %s", clausenames[i]);
	
		if (supported_dirs[dirtype].clauses[i].nmodifs)
		{
			fprintf(stderr, " [(");
			for (j = OCM_none+1; j < OCM_lastclmod; j++)
			{
				if (supported_dirs[dirtype].clauses[i].ocms[j])
				{
					mods++;
					fprintf(stderr, "%s: ", clausemods[j]);
					if (mods != supported_dirs[dirtype].clauses[i].nmodifs)
						fprintf(stderr, "| ");
				}
			}
			fprintf(stderr, ")]");
		}
		fprintf(stderr, "\n");
	}
	fprintf(stderr, "\n\n");
}


/**
 * Print all directives which support the given clause
 *
 * @param cltype The clause type
 */
void clr_print_clause_dirs(ompclt_e cltype)
{
	int i, maxw = 0, clnum;

	assert(rules_inited); /* rules should be already parsed */
	assert(cltype != OCNOCLAUSE && cltype < OCLASTCLAUSE); /* cltype must be valid */

	/* Pass 1: get max width */
	for (i = 1; i < DCLASTDIR; i++)
		if (strlen(ompdirnames[i]) > maxw)
			maxw = strlen(ompdirnames[i]);
	if (maxw > 50)
		maxw = 50;    /* cutoff */

	/* Pass 2: actual printout */
	fprintf(stderr, "----- directives with the '%s' clause -----\n", 
	                clausenames[cltype]);
	for (i = 1; i < DCLASTDIR; i++)
	{
		clnum = supported_dirs[i].clauses[cltype].maxtimes;
		if (!clnum)
			continue;

		fprintf(stderr, "  %-*s", maxw, ompdirnames[i]);
		if (clnum != -1)
			fprintf(stderr, "   // max %s clauses: %d",  clausenames[cltype], clnum);
		fprintf(stderr, "\n");
	}
	fprintf(stderr, "\n\n");
}


/**
 * Load all clause rules
 */
void clr_initrules(void)
{
	int i, j;

	if (rules_inited)
		return;
	for (i = 0; allrules[i] != NULL; i++)
		for (j = 0; (*allrules[i])[j].dirtype != DCNONE; j++)
			parse_directive_clrules((*allrules[i])[j].dirtype, 
				(*allrules[i])[j].clausestr);
	rules_inited = true;
}
