/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* cfg.h -- control flow graph module */

#ifndef __CFG_H__
#define __CFG_H__

#include <stdbool.h>
#include "ast.h"
#include "set.h"

/* 4 types of nodes in the CFG.
 * Joiners are temporary nodes used when constructing the CFG out of the AST.
 * They can be removed with cfg_remove_joiners().
 * Block nodes represent basic blocks (if basic blocks get generated)
 * All other nodes are plain.
 */
typedef enum
{
	CFG_TERMINAL, CFG_JOINER, CFG_BLOCK, CFG_PLAIN
} cfgtype_e;

/* Each plain node points to the AST node it came from.
 * The AST node will be either a statement or an expression.
 */
typedef enum { STMTNODE, EXPRNODE } asttype_e;

/* Helper nodes are the terminal ones plus the joiners */
#define CFG_HELPER_NODE(n) ((n)->type ==CFG_TERMINAL || (n)->type ==CFG_JOINER)

/* A CFG node
 */
typedef struct cfgnode_ *cfgnode;
struct cfgnode_
{
	cfgtype_e type;
	asttype_e astnodetype; /* What type of AST node it came from */
	aststmt   stmt;        /* pointer to the AST node */
	astexpr   expr;        /* pointer to the expression part of the AST node */
	cfgnode   *succs;      /* successors */
	int       nsucc;       /* out-degree (#successors) */
	cfgnode   *backs;      /* back edges or predecessors (size = npred) */
	int       npred;       /* in-degree (# predecessors) */
	cfgnode   cfgptr;      /* auxiliary info; bblocks use this */
	cfgnode   stacknext;   /* for stack participation */
};


SET_TYPE_DEFINE(cfg, cfgnode, cfgnode, 1031)
SET_TYPE_DEFINE(lbl, symbol, cfgnode, 1031)


extern void cfg_test(aststmt tree, int mode);
extern set(cfg) cfg_create(aststmt tree, int mode);
extern bool cfg_is_stmt_cutnode(aststmt tree, set(cfg) cutnodes);

#endif
