/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_xformrules.c -- transformation rules */

/* This unit registers transformation functions for each OpenMP construct.
 * In addition, it defines the handling policy (split/combine) for combined,
 * composite and exlusively nested(*) constructs.
 * 
 * (*) We call two constructs A and B *exclusively nested* if B is closely 
 * nested within A and there is no other statement nested in A, e.g.
 * #pragma omp parallel            // A
 * {
 *   #pragma omp for               // B
 *   for (i=0;...) { ... }
 * } 
 * 
 * See below how to add customized rules for modules/devices.
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "ompi.h"
#include "builder.h"
#include "autoscope.h"
#include "set.h"
#include "ast_xform.h"
#include "ast_xformrules.h"
#include "ast_copy.h"
#include "x_clauses.h"
#include "x_clauserules.h"
#include "x_combine.h"
#include "x_parallel.h"
#include "x_task.h"
#include "x_single.h"
#include "x_sections.h"
#include "x_loops.h"
#include "x_for.h"
#include "x_thrpriv.h"
#include "x_shglob.h"
#include "x_target.h"
#include "x_kernels.h"
#include "x_decltarg.h"
#include "x_cars.h"
#include "x_teams.h"
#include "x_arrays.h"
#include "x_distribute.h"
#include "x_simd.h"
#include "x_ordered.h"
#include "x_atomic.h"
#include "x_cancel.h"


/* Types and macros
 */
typedef struct xfr_rule_
{
	ompdirt_e dirtype;
	void (*xform_func)(aststmt *t);
	int combsplittype;
	char *copyfrom;
} xfr_rule_t;

typedef struct xfr_mod_
{
	char *modulename;
	xfr_rule_t **rules;
	xfr_vars_t vars;
} xfr_mod_t;

#define XFR_DEFINE(d)        struct xfr_vars_ d##_xformvars; \
                             xfr_rule_t *d##_xforminfo = (xfr_rule_t[])
#define XFR_DEFINE_COPY(d,c) struct xfr_vars_ d##_xformvars; \
                             xfr_rule_t *d##_xforminfo = (xfr_rule_t[]) \
                               { { DCNONE, (void *) 0, 0, #c } };

#define XFR_MOD(d)           { #d, &d##_xforminfo, &d##_xformvars }
#define XFR_LASTMOD          { "", (xfr_rule_t**) 0, (xfr_vars_t) 0 }
#define XFR_LASTRULE         { DCNONE, (void*) 0, 0, 0 }
#define XFR_DEFINE_DEFAULT   XFR_DEFINE(_)
#define DEFAULT_XFR_MOD      XFR_MOD(_)

#define XFR_GET(x)           (all_xformrules[x])
#define XFR_RULES(x)         (XFR_GET(x).rules)
#define XFR_DEVNAME(x)       (XFR_GET(x).modulename)
#define XFR_VARS(x)          (XFR_GET(x).vars)
#define XFR_GETVAR(x,v)      (XFR_VARS(x)->v)

#define XFR_IS(x,y)          (!strcmp(XFR_DEVNAME(x), y))
#define XFR_ISLAST(x)        ((!strcmp(XFR_DEVNAME(x), "")) && \
                              (XFR_RULES(x) == NULL))

#define XFR_GETRULE(x,y)     ((*XFR_RULES(x))[y])
#define XFR_GETRULETYPE(x,y) (XFR_GETRULE(x,y).dirtype)
#define XFR_GETCOPY(x,y)     (XFR_GETRULE(x,y).copyfrom)

#define XFR_ISLASTRULE(x,y)  (XFR_GETRULETYPE(x,y) == DCNONE)
#define XFR_ISCOPY(x,y)      ((XFR_GETRULETYPE(x,y) == DCNONE) && \
                              (XFR_GETCOPY(x,y) != NULL))


/* The DEFAULT rules apply to the host and, in absence of other rules,
 * to the code targeting all modules/devices.
 * 
 * To define new, module-specific transformation rules for some constructs,
 * use the following rules syntax:
 * 
 * XFR_DEFINE(<modulename>) {
 *     { <constuct-type>, <xform-function>, <combine/split-policy> },
 *     ...,
 *     XFR_LASTRULE       // This ends the new transformation rules
 * }
 * 
 * Then include the new transformation rules to the global collection:
 * 
 * xfr_mod_t all_xformrules[] = {
 *     DEFAULT_XFR_MOD,   // Should always be first
 *     ...,
 *     XFR_MOD(<modulename>),
 *     XFR_LASTMOD        // Should always be last
 * }
 * 
 * Important notes:
 * 
 *   Directives that have been assigned with a policy different
 *   from NONE, will be handled by the transformation function of the
 *   resulted directive.
 *   
 * Examples:
 * (a) #pragma omp target parallel for 
 * 
 *     with SPLITONLY gets transformed to:
 * 
 *     #pragma omp target 
 *       #pragma omp parallel for
 *   
 *     and handled as a `#pragma omp target'
 *     (i.e. the corresponding DCTARGET xform function will be called)
 * 
 * (b) #pragma omp target teams distribute
 *       #pragma omp parallel for
 * 
 *     with BOTH policy gets transformed to:
 * 
 *     (1) #pragma omp target teams distribute parallel for
 *  
 *     then to:
 *  
 *     (2) #pragma omp target teams
 *           #pragma omp distribute parallel for
 *    
 *     and gets handled as a `#pragma omp target teams'
 *     (i.e. the corresponding DCTARGETTEAMS xform function will be called)
 */

XFR_DEFINE_DEFAULT {
	{ DCTARGET,                    xform_target_default,      XFR_NONE },
	{ DCTARGETUPD,                 xform_targetupdate,        XFR_NONE },
	{ DCTARGENTERDATA,             xform_targetenterdata,     XFR_NONE },
	{ DCTARGEXITDATA,              xform_targetexitdata,      XFR_NONE },
	{ DCTARGETPARALLEL,            xform_targetparallel,      XFR_SPLITONLY },
	{ DCTARGETPARFOR,              xform_targparfor_default,  XFR_SPLITONLY },
	{ DCTARGETTEAMS,               xform_targetteams_default, XFR_SPLITONLY },
	{ DCTARGETTEAMSDIST,           xform_targetteams_default, XFR_SPLITONLY },
	{ DCTARGETTEAMSDISTSIMD,       xform_targetteamsdistsimd, XFR_NONE },
	{ DCTARGETTEAMSDISTPARFOR,     xform_targetteams_default, XFR_SPLITONLY },
	{ DCTARGETTEAMSDISTPARFORSIMD, xform_targetteamsdistparforsimd, XFR_NONE },
	{ DCTEAMS,                     xform_teams,               XFR_NONE },
	{ DCTEAMSDISTSIMD,             xform_teamsdistsimd,       XFR_NONE },
	{ DCTEAMSDISTPARFOR,           xform_teams,               XFR_SPLITONLY },
	{ DCTEAMSDISTPARFORSIMD,       xform_teamsdistparforsimd, XFR_NONE },
	{ DCPARALLEL,                  xform_parallel_wrapper,    XFR_NONE },
	{ DCPARSECTIONS,               xform_parallel_wrapper,    XFR_SPLITONLY },
	{ DCPARFOR,                    xform_parallel_wrapper,    XFR_SPLITONLY },
	{ DCPARFORSIMD,                xform_parforsimd,          XFR_NONE },
	{ DCDISTSIMD,                  xform_distsimd,            XFR_NONE },
	{ DCDISTPARFORSIMD,            xform_distparforsimd,      XFR_NONE },
	{ DCFORSIMD,                   xform_forsimd,             XFR_NONE },
	{ DCSIMD,                      xform_simd,                XFR_NONE },
	{ DCCANCEL,                    xform_cancel,              XFR_NONE },
	{ DCCANCELLATIONPOINT,         xform_cancellationpoint,   XFR_NONE },
	{ DCTHREADPRIVATE,             xform_threadprivate,       XFR_NONE },  
	{ DCATOMIC,                    xform_atomic,              XFR_NONE },
	{ DCBARRIER,                   xform_barrier,             XFR_NONE },
	{ DCTASKWAIT,                  xform_taskwait,            XFR_NONE },
	{ DCTASKYIELD,                 xform_taskyield,           XFR_NONE },
	{ DCFLUSH,                     xform_flush,               XFR_NONE },
	{ DCCRITICAL,                  xform_critical,            XFR_NONE },
	{ DCMASTER,                    xform_master,              XFR_NONE },
	{ DCORDERED,                   xform_ordered,             XFR_NONE },
	{ DCSINGLE,                    xform_single,              XFR_NONE },
	{ DCSECTIONS,                  xform_sections,            XFR_NONE },
	{ DCFOR_P,                     xform_for,                 XFR_NONE },
	{ DCFOR,                       xform_for,                 XFR_NONE },
	{ DCTASK,                      __wrap_xform_task,         XFR_NONE },
	XFR_LASTRULE
};

/* Transformation rules for CUDA devices */
XFR_DEFINE(cuda) {
	{ DCTARGET,                xform_target_cuda,             XFR_COMBINEONLY },
	{ DCTARGETPARALLEL,        xform_targetteams_cuda,        XFR_BOTH },
	{ DCTARGETPARFOR,          xform_targparfor_cuda,         XFR_SPLITONLY },
	{ DCTARGETTEAMS,           xform_targetteams_cuda,        XFR_NONE },
	{ DCTARGETTEAMSDIST,       xform_targetteams_cuda,        XFR_BOTH },
	{ DCTARGETTEAMSDISTPARFOR, xform_targetteamsdistparfor,   XFR_SPLITONLY },
	{ DCCRITICAL,              xform_critical_cuda,           XFR_NONE },
	{ DCATOMIC,                xform_atomic_cuda,             XFR_NONE },
	{ DCPARFOR,                xform_parallel_wrapper,        XFR_SPLITONLY },
	{ DCPARALLEL,              xform_parallel_cuda_wrapper,   XFR_NONE },
	{ DCTASK,                  __wrap_xform_task_cuda,        XFR_NONE },
	XFR_LASTRULE
};

/* The full rules collection */
xfr_mod_t all_xformrules[] = { 
	DEFAULT_XFR_MOD, 
	XFR_MOD(cuda),
	XFR_LASTMOD
};


/* Set keeping all the transformation rules. */
SET_TYPE_IMPLEMENT(xformrules)
set(xformrules) ast_xfrules;

/* Boolean set to `true' only when rules are successfully
 * parsed. 
 */
static bool ast_xfr_inited = false;

/**
 * Call the transformation function for a specific
 * directive, according to the transformation rules of
 * a specific module.
 * 
 * @param modulename The module name
 * @param t          The input tree
 */
void ast_xfr_call_xformfunc(char *modulename, aststmt *t)
{
	setelem(xformrules) e, defxfrules = ast_xfr_get_defaultrules();

	if (!ast_xfr_inited)
		exit_error(1, "[%s] xform rules not initialized; exiting.\n",
		"ast_xfr_call_xformfunc");

	assert((e = set_get(ast_xfrules, modulename)) != NULL);
	if (!e->value->rules[(*t)->u.omp->type].initialized)
		assert(defxfrules->value->rules[(*t)->u.omp->type].initialized);

	if (!e->value->rules[(*t)->u.omp->type].xform_func)
		defxfrules->value->rules[(*t)->u.omp->type].xform_func(t);
	else
		e->value->rules[(*t)->u.omp->type].xform_func(t);
}

/**
 * Parse the declared transformation rules.
 */
void ast_xfr_initrules()
{
	int i, j, realindex;
	setelem(xformrules) e;

	set_init(xformrules, &ast_xfrules);
	for (i = 0; !XFR_ISLAST(i); i++)
	{
		/* If the ruleset is a copy of another one, then
		 * use the index of the original set */
		realindex = XFR_ISCOPY(i,0)
			? set_get(ast_xfrules, XFR_GETCOPY(i,0))->value->origindex
			: i;
	
		e = set_put(ast_xfrules, XFR_DEVNAME(i));
		e->value = (xfr_modrules_t *) malloc(sizeof(xfr_modrules_t));
		e->value->origindex = i;
		e->value->vars = XFR_VARS(i);
		e->value->vars->newglobals = e->value->vars->targtree = NULL;
		e->value->vars->targetnum = 0;
		e->value->vars->Targets = NULL;
		for (j = 0; !XFR_ISLASTRULE(realindex, j); j++)
		{
			e->value->rules[XFR_GETRULETYPE(realindex, j)].initialized = true;
			e->value->rules[XFR_GETRULETYPE(realindex, j)].xform_func = 
				XFR_GETRULE(realindex, j).xform_func;
			e->value->rules[XFR_GETRULETYPE(realindex, j)].combsplittype = 
				XFR_GETRULE(realindex, j).combsplittype;
		}
	}

	ast_xfr_inited = true;
}

/**
 * Get the default transformation rules
 */
setelem(xformrules) ast_xfr_get_defaultrules()
{
	return set_get(ast_xfrules, DEFAULTDEVICE);
}

/**
 * Returns `true', if a directive type is supported
 * by the default transformation rules.
 * 
 * @param dirtype The directive type
 */
bool ast_xfr_type_issupported(ompdirt_e dirtype)
{
	setelem(xformrules) defxfrules = ast_xfr_get_defaultrules();
	return defxfrules->value->rules[dirtype].initialized;
}

/**
 * Generate N copies of an input tree, where N = the number
 * of different modules that have customized transformation
 * rules.
 *
 * @param t The input tree
 */
void ast_xfr_gen_tree_copies(aststmt *t)
{
	setelem(xformrules) e, defxfrules = ast_xfr_get_defaultrules();
	defxfrules->value->vars->tree = t;

	for (e = ast_xfrules->first; e; e = e->next)
	{
		if (e == defxfrules || !strstr(MODULES_CONFIG, e->key))
			continue;

		if (!e->value->rules[(*t)->u.omp->type].initialized)
			continue;

		e->value->vars->tree = (aststmt *) malloc(sizeof(aststmt));
		*(e->value->vars->tree) = ast_stmt_copy(*t);
		ast_stmt_parent((*t)->parent, *(e->value->vars->tree));
	}	
}

/**
 * Free all the generated tree copies for each set of
 * transformation rules
 */
void ast_xfr_free_tree_copies()
{
	setelem(xformrules) e, defxfrules = ast_xfr_get_defaultrules();

	for (e = ast_xfrules->first; e; e = e->next)
	{
		if (e == defxfrules || !strstr(MODULES_CONFIG, e->key))
			continue;
		free(e->value->vars->tree);
	}	
}
