/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_xform.c */

/*
 * 2010/10/20:
 *   fixed transformation omission of return statements.
 * 2009/12/10:
 *   thread/task functions now placed *after* the parent function so that
 *   recursion works without any warnings.
 * 2009/05/04:
 *   fixed unbelievable omission in master,ordered,atomic & critical constructs
 * 2007/09/04:
 *   brand new thread funcs handling
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "ompi.h"
#include "builder.h"
#include "autoscope.h"
#include "ast_copy.h"
#include "ast_free.h"
#include "ast_print.h"
#include "ast_vars.h"
#include "ast_xform.h"
#include "ast_types.h"
#include "ast_arith.h"
#include "ast_assorted.h"
#include "ast_xformrules.h"
#include "x_clauses.h"
#include "x_clauserules.h"
#include "x_combine.h"
#include "x_parallel.h"
#include "x_task.h"
#include "x_single.h"
#include "x_sections.h"
#include "x_loops.h"
#include "x_for.h"
#include "x_thrpriv.h"
#include "x_shglob.h"
#include "x_target.h"
#include "x_kernels.h"
#include "x_decltarg.h"
#include "x_cars.h"
#include "x_teams.h"
#include "x_arrays.h"
#include "x_distribute.h"
#include "x_simd.h"
#include "x_ordered.h"
#include "x_atomic.h"
#include "x_cancel.h"
#include "ox_xform.h"

/*
 * 2009/12/09:
 *   added forgotten type transforms in selection & iteration stmt expressions
 * 2009/05/03:
 *   fixed bug (no-xfrom) in barrier
 */


/* The scope level of the most recently encountered parallel construct
 */
int closest_parallel_scope = -1;

/* Used in ast_declare_varlist_vars for determining if a variable appears in a
 * target data clause in order to set isindevenv to true.
 */
int target_data_scope = -1;

/* Used in outline.c for setting isindevenv of functions and here to prevent
 * target pragmas inside declare target
 * -- made it an int to cover multiply #declared items in v45 (VVD)
 */
int inDeclTarget = 0;

/* Used for determining the label for the cancel parallel GOTOs
 */
int cur_parallel_line = 0;

/* Used for determining the label for the cancel taskgroup GOTOs
 */
int cur_taskgroup_line = 0;

/* Keeps track of combined directives 
 */
static int dist_combined = 0;

bool in_gpu_xform()
{
	bool ingpu = false;
	
	ingpu = (!stack_isempty(ompdirectives)) && (!strcmp(XFORM_CURR_DIRECTIVE->modulename, "cuda"));
	ingpu = ingpu || ingpufuncdef;

	return ingpu;
}

bool in_target()
{
	bool intarget = false;
	setelem(xformrules) e;

	for (e = ast_xfrules->first; e; e = e->next)
		intarget = intarget || (e->value->vars->targtree != NULL);
	
	return intarget;
}

bool is_target_construct(ompdirt_e type)
{
	return ((type == DCTARGETDATA) || (type == DCTARGET) || 
		(type == DCTARGETTEAMS) || (type == DCTARGETTEAMSDIST) ||
		(type == DCTARGETTEAMSDISTPARFOR) || (type == DCTARGETPARALLEL) || 
		(type == DCTARGETPARFOR));
}

static
bool is_target_related(ompdirt_e type)
{
	return ((is_target_construct(type)) || (type == DCTARGETUPD) || (type == DCTARGENTERDATA )
	|| (type == DCTARGEXITDATA));
}

/* Degenerate case: check whether an empty-bodied omp construct is given.
 * If so, replace it by a comment.
 */
static
int empty_bodied_omp(aststmt *t)
{
	if ((*t)->u.omp->body == NULL)
		return 0;
	if ((*t)->u.omp->body->type == COMPOUND && (*t)->u.omp->body->body == NULL)
	{
		char *conname = ompdirnames[(*t)->u.omp->type];
		ast_stmt_free(*t);
		*t = verbit("/* (l%d) empty %s construct removed */", (*t)->l, conname);
		return (1);
	}
	return (0);
}

/* This function decides if we should require the OMPi runtime
 * and/or the OMPi device functionality and adds the appropriate
 * calls to the constructor.
 *
 * See ompi.c for more details on this effort. 
 */
static
void add_reqfuncs()
{
	int __has_only_decltargprotos = __has_onlydecltarg && !declvars_exist();
	__needs_devices = __has_target || __has_devfuncs || declvars_exist();
	
	/* Require the OMPi runtime */
	if ((__has_omp && !__has_only_decltargprotos) || (__has_ompfuncs))
	{
		bld_autoinits_add(
			Expression(
				FunctionCall(
					IdentName("ort_required"),
					NULL
				)
			)
		);
	}
	
	/* Require the OMPi device functionality */
	if (__needs_devices)
	{
		bld_autoinits_add(
			Expression(
				FunctionCall(
					IdentName("ort_modules_required"),
					NULL
				)
			)
		);
	}
		
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     AST TRAVERSAL OF NON-OPENMP NODES                         *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* The only thing we do here is correctly start/end scopes and
 * declare all identifiers. Whenever we insert a symbol into the
 * symbol table, we record its specifiers/declarator so as to
 * find it later if needed.
 */
void ast_stmt_iteration_xform(aststmt *t)
{
	switch ((*t)->subtype)
	{
		case SFOR:
		{
			int newscope = false;
			
			if ((*t)->u.iteration.init != NULL)
			{
				if ((*t)->u.iteration.init->type == DECLARATION)
				{
					newscope = true;
					scope_start(stab);  /* New scope for the declared variable(s) */
				}
				ast_stmt_xform(&((*t)->u.iteration.init));
			}
			if ((*t)->u.iteration.cond != NULL)
				ast_expr_xform(&((*t)->u.iteration.cond));
			if ((*t)->u.iteration.incr != NULL)
				ast_expr_xform(&((*t)->u.iteration.incr));
			ast_stmt_xform(&((*t)->body));
			if (newscope)
				scope_end(stab);
			break;
		}
		case SWHILE:
			if ((*t)->u.iteration.cond != NULL)
				ast_expr_xform(&((*t)->u.iteration.cond));
			ast_stmt_xform(&((*t)->body));
			break;
		case SDO:
			if ((*t)->u.iteration.cond != NULL)
				ast_expr_xform(&((*t)->u.iteration.cond));
			ast_stmt_xform(&((*t)->body));
			break;
	}
}

void ast_stmt_selection_xform(aststmt *t)
{
	switch ((*t)->subtype)
	{
		case SSWITCH:
			ast_expr_xform(&((*t)->u.selection.cond));
			ast_stmt_xform(&((*t)->body));
			break;
		case SIF:
			ast_expr_xform(&((*t)->u.selection.cond));
			ast_stmt_xform(&((*t)->body));
			if ((*t)->u.selection.elsebody)
				ast_stmt_xform(&((*t)->u.selection.elsebody));
			break;
	}
}

void ast_stmt_labeled_xform(aststmt *t)
{
	ast_stmt_xform(&((*t)->body));
}

SET_TYPE_IMPLEMENT(fbstmts);
set(fbstmts) funcbodycp;
bool ingpufuncdef = false;

static char *currfuncname = "";

/* Only statement types that matter are considered. */
void ast_stmt_xform(aststmt *t)
{
	if (t == NULL || *t == NULL) return;

	switch ((*t)->type)
	{
		case EXPRESSION:
			ast_expr_xform(&((*t)->u.expr));
			break;
		case ITERATION:
			ast_stmt_iteration_xform(t);
			break;
		case JUMP:   /* 2010-10-20 (PEH) */
			if ((*t)->subtype == SRETURN && (*t)->u.expr != NULL)
				ast_expr_xform(&((*t)->u.expr));
			break;
		case SELECTION:
			ast_stmt_selection_xform(t);
			break;
		case LABELED:
			ast_stmt_labeled_xform(t);
			break;
		case COMPOUND:
			if ((*t)->body)
			{
				scope_start(stab);
				ast_stmt_xform(&((*t)->body));
				scope_end(stab);
			}
			break;
		case STATEMENTLIST:
			ast_stmt_xform(&((*t)->u.next));
			ast_stmt_xform(&((*t)->body));
			break;
		case DECLARATION:
			ast_xt_declaration_xform(t);   /* transform & declare */
			break;
		case FUNCDEF:
		{
			symbol  fsym = decl_getidentifier_symbol((*t)->u.declaration.decl);
			stentry f    = symtab_get(stab, fsym, FUNCNAME);
			aststmt gpuizedbody;
			currfuncname = fsym->name;

			assert(f == NULL || f->funcdef == NULL); /* should not be def'ed twice */
			
			/* First declare the function & store its definition stmt */
			if (f == NULL)
			{
				f = symtab_put(stab,fsym, FUNCNAME);
				f->spec = (*t)->u.declaration.spec;
				f->decl = (*t)->u.declaration.decl;
			}
			f->funcdef = *t;
			
			if (decltarg_id_isknown(fsym)) /* Note that we have a #declare function */
			{
				inDeclTarget++;
				decltarg_bind_id(f);  /* Mark it, too */
			}
			
			scope_start(stab);         /* New scope here */

			if ((*t)->u.declaration.dlist) /* Old style */
			{
				/* Take care of array params *before* declaring them */
				ast_xt_dlist_array2pointer((*t)->u.declaration.dlist);  /* !!array params */
				ast_stmt_xform(&((*t)->u.declaration.dlist));  /* declared themselves */
			}
			else                           /* Normal; has paramtypelist */
			{
				ast_xt_barebones_substitute(&((*t)->u.declaration.spec),
										&((*t)->u.declaration.decl));
				/* Take care of array params *before* declaring them */
				ast_xt_decl_array2pointer((*t)->u.declaration.decl->decl->u.params);
				ast_declare_function_params((*t)->u.declaration.decl);/* decl manualy */
			}

			/* Store a gpuized+xformed replacement of the function's body
			 * (useful for cuda devices) */
			if (!ingpufuncdef 
				&& set_get(tarfun, fsym->name) 
				&& strcmp(fsym->name, MAIN_NEWNAME))
			{
				ingpufuncdef = true;
				gpuizedbody = (*t)->body ? ast_stmt_copy((*t)->body) : NULL;
				ingpufuncdef = false;
			}

			ast_stmt_xform(&((*t)->body));
			tp_fix_funcbody_gtpvars((*t)->body);    /* take care of gtp vars */
	
			/* The GPUized function body must be transformed a single time.
			 * We also store it in a specific set, which will be traversed
			 * later by x_kernels.c, for the actual function replacement.
			 */
			if (!ingpufuncdef 
				&& set_get(tarfun, fsym->name) 
				&& strcmp(fsym->name, MAIN_NEWNAME))
			{
				ingpufuncdef = true;
		
				#if PARALLEL_SCHEME == SCHEME_IFMASTER
				ifmaster_stmt(&gpuizedbody, true);
				#endif

				ast_stmt_parent((*t), gpuizedbody);
				ast_stmt_xform(&gpuizedbody);
				set_put(funcbodycp, fsym->name)->value = gpuizedbody;
				tp_fix_funcbody_gtpvars(gpuizedbody);
				ingpufuncdef = false;
			}

			if (processmode)
				analyze_pointerize_sgl(*t);        /* just replace them with pointers */

			scope_end(stab);           /* Scope done! */
			
			if (decltarg_id_isknown(fsym))  /* #declare function no more */
				inDeclTarget--;
			break;
		}
		case ASMSTMT:
			break;
		case OMPSTMT:
			if (enableOpenMP || testingmode) ast_omp_xform(t);
			break;
		case OX_STMT:
			if (enableOmpix || testingmode) ast_ompix_xform(t);
			break;
		default:
			break;
	}
}


/* Expressions are only examined for type casts and sizeof(type) */
void ast_expr_xform(astexpr *t)
{
	if (t == NULL || *t == NULL) return;
	switch ((*t)->type)
	{
		case CONDEXPR:
			ast_expr_xform(&((*t)->u.cond));
		/* Then continue below */
		case IDENT:
		case FUNCCALL:
		case BOP:
		case ASS:
		case DESIGNATED:
		case COMMALIST:
		case SPACELIST:
		case ARRAYIDX:
			if ((*t)->right)
				ast_expr_xform(&((*t)->right));
		/* Then continue below */
		case DOTFIELD:
		case PTRFIELD:
		case BRACEDINIT:
		case PREOP:
		case POSTOP:
		case IDXDES:
			ast_expr_xform(&((*t)->left));
			break;
		case CASTEXPR:
			ast_xt_barebones_substitute(&((*t)->u.dtype->spec), &((*t)->u.dtype->decl));
			ast_expr_xform(&((*t)->left));
			break;
		case UOP:
			if ((*t)->opid == UOP_sizeoftype || (*t)->opid == UOP_typetrick)
				ast_xt_barebones_substitute(&((*t)->u.dtype->spec), &((*t)->u.dtype->decl));
			else
				ast_expr_xform(&((*t)->left));
			break;
		default:
			break;
	}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     OPENMP NODES TO BE TRANSFORMED                            *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


void ast_ompcon_xform(ompcon *t)
{
	if ((*t)->body)     /* barrier & flush don't have a body! */
		ast_stmt_xform(&((*t)->body));
}

/**
 * If the original array was declared with its 1st dimension size missing,
 * then any copy will be problematic because it will not have an initializer
 * from which to discover the size. This function fixes it by forcing a
 * size based on the cardinality of the original array initializer.
 * 
 * @param copy  A clone of the original array declaration
 * @param orig  The original array declaration (with initializer)
 * @return      The clone (copy), with modified 1st dimension
 */
static
astdecl fix_array_decl_copy(astdecl copy, astdecl orig)
{
	/* arr_dimension_size(copy, 0, NULL) is assumed to be NULL */
	int ic = decl_initializer_cardinality(orig);
	
	if (ic == 0)
		exit_error(2, "(%s, line %d): cannot determine missing dimension "
					  "size of '%s'\n", orig->file->name, orig->l, 
					  decl_getidentifier_symbol(orig)->name);
	arr_dimension_size(copy, 0, numConstant(ic));
	return (copy);
}


/**
 * Reproduces only the Declarator part of the original declaration; the
 * special thing about this is that it takes care of arrays with missing
 * size for the 1st dimension.
 * 
 * @param e  The symtab entry for the var whose declaratοr we want to reproduce
 * @return   A declaration node (astdecl)
 */
astdecl xform_clone_declonly(stentry e)
{
	astdecl decl = ast_decl_copy(e->decl);
	
	if (e->isarray && arr_dimension_size(decl, 0, NULL) == NULL)
	{
		if (e->idecl)
			fix_array_decl_copy(decl, e->idecl);
		else
			exit_error(2, "(%s, line %d): '%s' cannot be cloned; "
						  "unknown dimension size\n", 
						  e->decl->file->name, e->decl->l, e->key->name);
	}
	return (decl);
}


/**
 * Reproduces the original declaration (optionally with a new variable name)
 * which may optionally have an initializer and/or convert the variable into 
 * a pointer.
 *
 * @todo "const" is not reproduced giving a warning
 *
 * @param s       The variable whose declaration we want to reproduce
 * @param initer  An optional initializer
 * @param mkptr   If true the variable will be declared as a pointer
 * @param snew    An optional new name for the variable
 * @return        A statement with the declaration
 */
inline aststmt 
xform_clone_declaration(symbol s, astexpr initer, bool mkptr, symbol snew)
{
	stentry e    = symtab_get(stab, s, IDNAME);
	astdecl decl = xform_clone_declonly(e);

	if (snew)     /* Change the name */
	{
		astdecl id = IdentifierDecl(snew);
		*(decl_getidentifier(decl)) = *id;
		free(id);
	}
	if (mkptr)    /* Turn to pointer */
		decl = decl_topointer(decl);
	return Declaration(
			 ast_spec_copy_nosc(e->spec),
			 initer ? InitDecl(decl, initer) : decl
		   );
}


/**
 * Reproduces the original declaration of a function
 *
 * @param funcname  The function whose declaration we want to reproduce
 * @return          A statement with the declaration
 */
inline aststmt xform_clone_funcdecl(symbol funcname)
{
	stentry func = symtab_get(stab, funcname, FUNCNAME);
	assert(func != NULL);
	return (Declaration(ast_spec_copy(func->spec), ast_decl_copy(func->decl)));
}


/* This one checks whether an openmp construct should include an
 * implicit barrier. This is needed only if the nowait is NOT present.
 * It tries to avoid the barrier by checking if it is redundant:
 *    if the construct is the last statement in the body of another
 *    construct which ends there and does have a barrier, then we don't
 *    need to output one more here.
 */
int xform_implicit_barrier_is_needed(ompcon t)
{
	aststmt p = t->parent;    /* the openmp statement */

	/* We must go up the tree to find the closest BlockList ancestor who has
	 * us as the rightmost leaf (i.e. last statement).
	 */
	for (; p->parent->type == STATEMENTLIST && p->parent->body == p; p=p->parent)
		;
	/* Now, climb up the compounds */
	for (; p->parent->type == COMPOUND; p = p->parent)
		;
	/* We may be in luck only if p is an openmp statement */
	if ((p = p->parent)->type == OMPSTMT)
	{
		/* Now this statement must be a convenient & blocking one; loops won't do */
		t = p->u.omp;
		if (t->type == DCPARSECTIONS || t->type == DCSECTIONS ||
			t->type == DCPARALLEL || t->type == DCSINGLE)
			if (xc_ompcon_get_clause(t, OCNOWAIT) == NULL)
				return (0);  /* Not needed after all! */
	}
	return (1);        /* Needed */
}


void xform_master(aststmt *t)
{
	aststmt s = (*t)->u.omp->body, parent = (*t)->parent, v;
	bool    parisif;

	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */

	/* Check if our parent is an IF statement */
	parisif = (*t)->parent->type == SELECTION && (*t)->parent->subtype == SIF;

	(*t)->u.omp->body = NULL;     /* Make it NULL so as to free t easily */
	ast_free(*t);                 /* Get rid of the OmpStmt */

	/* Here we play a bit risky: this might be a single statement, and we
	 * convert it to a block list; it seems that it won't create problems;
	 * should we better turn it into a compound?
	 */
	/* Explanation of the following code:
	 * We output an "if { ... }" thing. If our parent is an IF statement,
	 * we should enclose the whole thing in curly braces { } (i.e. create a
	 * compound) so as not to cause syntactic problems with a possibly
	 * following "else" statement.
	 */
	*t = Block3(
		   v,
		   If( /* omp_get_thread_num() == 0 */
			 BinaryOperator(
			   BOP_eqeq,
			   Call0_expr("omp_get_thread_num"),
			   ZeroExpr()
			 ),
			 s,
			 NULL
		   ),
		   linepragma(s->l + 1 - parisif, s->file)
		 );
	if (parisif)
		*t = Compound(*t);
	ast_stmt_parent(parent, *t);
}


/* In many transforms we use an "stlist" flag. This is to
 * check if our parent is a statement list etc so as to
 * avoid enclosing the produced code into a compound statement.
 * If one does not care for the looks, the produced code
 * should *always* be enlosed into a compound statement.
 */

void xform_critical(aststmt *t)
{
	aststmt s = (*t)->u.omp->body, parent = (*t)->parent, v;
	char    lock[128];
	bool    stlist;   /* See comment above */

	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	stlist = ((*t)->parent->type == STATEMENTLIST ||
	          (*t)->parent->type == COMPOUND);

	/* A lock named after the region name */
	if ((*t)->u.omp->directive->u.region)
		snprintf(lock,127,"_ompi_crity_%s", (*t)->u.omp->directive->u.region->name);
	else
		strcpy(lock, "_ompi_crity");

	(*t)->u.omp->body = NULL;    /* Make it NULL so as to free it easily */
	ast_free(*t);                /* Get rid of the OmpStmt */

	if (!symbol_exists(lock))    /* Check for the lock */
		bld_globalvar_add(Declaration(Declspec(SPEC_void), /* Dont use omp_lock_t */
		                         InitDecl(
		                           Declarator(
		                             Pointer(),
		                             IdentifierDecl(Symbol(lock))
		                           ),
		                           NullExpr()
		                         )));

	*t = Block5(
		v,
		FuncCallStmt(   /* ort_critical_begin(&lock); */
			IdentName("ort_critical_begin"),
			UOAddress(IdentName(lock))
		),
		s,
		FuncCallStmt(   /* ort_critical_end(&lock); */
			IdentName("ort_critical_end"),
			UOAddress(IdentName(lock))
		),
		linepragma(s->l + 1 - (!stlist), s->file)
		);

	if (!stlist)
		*t = Compound(*t);
	ast_stmt_parent(parent, *t);
}


void xform_critical_cuda(aststmt *t)
{
	aststmt s = (*t)->u.omp->body, parent = (*t)->parent, v;
	char    lock[128], blocked[128], shlock[128];
	bool    stlist;   /* See comment above */

	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	stlist = ((*t)->parent->type == STATEMENTLIST ||
	          (*t)->parent->type == COMPOUND);

	/* A lock named after the region name */
	if ((*t)->u.omp->directive->u.region)
	{
		snprintf(shlock,127,"_sh_ompi_crity_%s", (*t)->u.omp->directive->u.region->name);
		snprintf(lock,127,"_ompi_crity_%s", (*t)->u.omp->directive->u.region->name);
		snprintf(blocked,127,"_ompi_crity_blocked_%s", (*t)->u.omp->directive->u.region->name);
	}
	else
	{
		strcpy(shlock, "_sh_ompi_crity");
		strcpy(lock, "_ompi_crity");
		strcpy(blocked, "_ompi_crity_blocked");
	}

	(*t)->u.omp->body = NULL;    /* Make it NULL so as to free it easily */
	ast_free(*t);                /* Get rid of the OmpStmt */
						 
	*t = Block3(
		v,
        Block4(
            Declaration(
                Declspec(SPEC_int), 
                InitDecl(
                    Declarator(
                        Declspec(SPEC_star), 
                        IdentifierDecl(Symbol(shlock))
                    ), 
                    FunctionCall(IdentName("__cudadev_get_criticallock"), NULL)
                )
            ),
			FuncCallStmt(   /* ort_critical_begin(&lock); */
				IdentName("ort_critical_begin"),
				UOAddress(IdentName(lock))
			),
			Block3(
                Expression(
                    FunctionCall(IdentName("__cudadev_set_lock"), 
                    UnaryOperator(UOP_addr, IdentName(shlock)))
                ),
                s,
                Expression(
                    FunctionCall(IdentName("__cudadev_unset_lock"), 
                    UnaryOperator(UOP_addr, IdentName(shlock)))
                )
            ),
			FuncCallStmt(   /* ort_critical_end(&lock); */
                IdentName("ort_critical_end"),
                UOAddress(IdentName(lock))
			)
		),
		linepragma(s->l + 1 - (!stlist), s->file)
		);

	if (!stlist)
		*t = Compound(*t);
	ast_stmt_parent(parent, *t);
}


void xform_taskgroup(aststmt *t)
{
	aststmt s = (*t)->u.omp->body, parent = (*t)->parent, v;
	bool    stlist;   /* See comment above */
	char    clabel[23];

	snprintf(clabel, 23, "CANCEL_taskgroup_%d", cur_taskgroup_line);
	
	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	stlist = ((*t)->parent->type == STATEMENTLIST ||
	          (*t)->parent->type == COMPOUND);

	(*t)->u.omp->body = NULL;    /* Make it NULL so as to free it easily */
	ast_free(*t);                /* Get rid of the OmpStmt */

	*t = Block5(
	       v,
	       Call0_stmt("ort_entering_taskgroup"),
	       s,
	       Labeled(
	         Symbol(clabel), /* label used for cancel */
	         Call0_stmt("ort_leaving_taskgroup")
	       ),
	       linepragma(s->l + 1 - (!stlist), s->file)
	     );

	if (!stlist)
		*t = Compound(*t);
	ast_stmt_parent(parent, *t);
}

void xform_flush(aststmt *t)
{
	aststmt parent = (*t)->parent, v;

	v = ompdir_commented((*t)->u.omp->directive);  /* Put directive in comments */
	ast_free(*t);                                  /* Cut the tree here */
	/* flush occurs ONLY as a blocklist */
	*t = BlockList(v, Call0_stmt("ort_fence"));
	ast_stmt_parent(parent, *t);                   /* This MUST BE DONE!! */
}

/* Used to be a macro in ast_xform.h. Was turned into a function to incorporate
 * cancel logic.
 */
aststmt BarrierCall()
{
	char label[23], mabel[23];
	
	if (!cur_parallel_line && !cur_taskgroup_line)
		return Call0_stmt("ort_barrier_me");
	if (cur_parallel_line && !cur_taskgroup_line)
	{
		snprintf(label, 23, "CANCEL_parallel_%d", cur_parallel_line);
		/* if(ort_barrier_me()) goto <label>; */
		return If(BinaryOperator(BOP_eqeq, 
	                             Call0_expr("ort_barrier_me"), OneExpr()), 
	              Goto(Symbol(label)), 
	              NULL);
	}
	if (!cur_parallel_line && cur_taskgroup_line)
	{
		snprintf(label, 23, "CANCEL_taskgroup_%d", cur_taskgroup_line);
		/* if(ort_barrier_me()) goto <label>; */
		return If(BinaryOperator(BOP_eqeq, 
	                             Call0_expr("ort_barrier_me"), numConstant(2)), 
	              Goto(Symbol(label)), 
	              NULL);
	}
	/* 4th case: both parallel and taskgroup constructs active */
	snprintf(label, 23, "CANCEL_parallel_%d", cur_parallel_line);
	snprintf(mabel, 23, "CANCEL_taskgroup_%d", cur_taskgroup_line);
	return Switch(Call0_expr("ort_barrier_me"),
	         Compound(
	           Block3(
	             Case(ZeroExpr(), Break()), 
	             Case(OneExpr(), Goto(Symbol(label))),
	             Case(numConstant(2), Goto(Symbol(mabel)))
	           )
	         )
	       );
}

void xform_barrier(aststmt *t)
{
	aststmt parent = (*t)->parent, v;

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);
	*t = BlockList(v, BarrierCall());
	ast_stmt_parent(parent, *t);
}


void xform_taskwait(aststmt *t)
{
	aststmt parent = (*t)->parent, v;

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);
	*t = BlockList(v, FuncCallStmt(IdentName("ort_taskwait"),
	                               ZeroExpr()));
	ast_stmt_parent(parent, *t);
}


void xform_taskyield(aststmt *t)
{
	aststmt parent = (*t)->parent, v;

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);
	*t = v;            /* Just ignore it for now */

	ast_stmt_parent(parent, *t);
}

/* See below for comments regarding the following func, which declares 
 * fully all private-scope vars.
 */
static
void declare_private_dataclause_vars(ompclause t)
{
	if (t == NULL) return;
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			declare_private_dataclause_vars(t->u.list.next);
		assert((t = t->u.list.elem) != NULL);
	}
	switch (t->type)
	{
		case OCPRIVATE:
		case OCFIRSTPRIVATE:
		case OCLASTPRIVATE:
		case OCCOPYPRIVATE:
		case OCCOPYIN:
			ast_declare_varlist_vars(t->u.varlist, t->type, t->subtype);
			break;
		case OCREDUCTION:
		case OCMAP:
			ast_declare_xlist_vars(t);
			break;
		default:
			break;
	}
}


/**
 * We must first transform the body of the construct and then
 * do the transformation of the construct itself (i.e. create new code
 * based on the clauses).
 * BUT, FOR CONSTRUCTS WITH DATA CLAUSES:
 *   before transforming the body, we have to *declare* (fully)
 *   all the identifiers in the clauses that have *private* scope,
 *   so that in the body those variables have the correct visibility.
 *   (the others don't need any treatment as they have already been
 *    declared in previous scopes).
 *
 * Thus, we begin our new scope, declare them, transform the body
 * and finaly close the scope (ditching the declarations).
 * Note that the declarations, DO NOT PRODUCE ANY NEW CODE; they
 * are simply inserted in the symbol table for the correct visibility
 * of variables used in the body. The declarations are produced when
 * we transform the construct itself.
 * @param t the openmp statement
 */
void xform_ompcon_body(ompcon t)
{
	if (t->body->type == COMPOUND && t->body->body == NULL)
		return;      /* degenerate case */
	
	xc_validate_only_dataclause_vars(t->directive);  /* check for duplicates */
	scope_start(stab);

	/* If we are in a target data put __ort_denv in the symbol table, so that if
	 * there is any clause that uses outline in the body it will put __ort_denv
	 * in the struct it creates.
	 */
	if (is_target_related(t->type))
	{
		aststmt tmp = get_denv_var_decl(true);
		ast_declare_decllist_vars(tmp->u.declaration.spec, tmp->u.declaration.decl);
	}

	declare_private_dataclause_vars(t->directive->clauses);
	ast_stmt_xform(&(t->body));     /* First transform the body */
	scope_end(stab);
}

STACK_TYPE_IMPLEMENT(ompdirs);
stack(ompdirs) ompdirectives;

static
void split_combine_xform(setelem(xformrules) e, ompdirt_e type, aststmt *t)
{
	ompdiritem_t con;
	int splcombtype;
	int iscombpar = 0;
	cur_parallel_line = cur_taskgroup_line = 0;	

	if (e == NULL)
	{
		fprintf(stderr, "[split_combine_xform] no rules for directive `%s'; exiting.", 
			ompdirnames[type]);
		exit(1);
	}
	
	/* (1) Find if the construct must be splitted or combined.
	 * Act accordingly */
	splcombtype = e->value->rules[type].combsplittype;
	switch (splcombtype)
	{
		case XFR_BOTH:
			ccc_try_combining(t);
			ccc_try_splitting(t);
			break;
		case XFR_SPLITONLY:
			if (type == DCPARFOR || type == DCPARSECTIONS) iscombpar = 1;
			ccc_try_splitting(t);
			break;
		case XFR_COMBINEONLY:
			ccc_try_combining(t);
			break;
		case XFR_NONE:
		default:
			break;
	}
	
	/* (2) Initialize and push a stack item containing information
	about the construct */
	con = malloc(sizeof(struct ompdiritem_));
	con->dirtype = (*t)->u.omp->type;
	con->modulename = e->key;
	con->prevdirtype = type;
	con->nparallel = 0;
	con->ismasterworker = 0;
	con->iscombpar = iscombpar;

	/* (3) Push the construct into the stack */
	stack_push(ompdirectives, con);

	if (is_target_construct(type))
		TARGTREE_INIT(e->value->vars->targtree);
	
	/* (4) Call the transformation function of the RESULTED type
	 *     (not the original one) */
	ast_xfr_call_xformfunc(e->key, t);
	
	/* (5) Pop the construct from the stack */
	stack_pop(ompdirectives);

	/* (6) Target constructs require freeing of their inner tree */
	if (is_target_construct(type))
		targtree_reset_all();
}

/**
 * Searches for an outer target construct in
 * the current construct stack
 *
 * @return The target stack element
 */
stackelem(ompdirs) find_target_parent()
{
	stackelem(ompdirs) e;
	for (e = ompdirectives->top; e; e = e->prev)
		if (e->data->dirtype == DCTARGET 
			|| e->data->dirtype == DCTARGETPARALLEL 
			|| e->data->dirtype == DCTARGETPARFOR)
			return e;
	return NULL;
}

/**
 * Searches for a perfectly nested parallel region 
 * 
 * @param t          The OpenMP construct, the body of which will be searched
 * @param dircontype The type of construct to search for
 * @return           The level of the discovered parallel region, otherwise 0
 */
int search_nested_construct(ompcon t, ompdirt_e dircontype)
{
	ompcon *tptr = &t;
	int level = 1;

	if (t == NULL) return 0;

	while ((*tptr)->type != dircontype)
	{
		if (dircontype == DCPARALLEL && (*tptr)->type == DCPARFOR) break;
		if ((*tptr == NULL) || ((*tptr)->body == NULL)) return 0;
		if ((*tptr)->body->type == COMPOUND) 
		{
			if ((*tptr)->body->body && (*tptr)->body->body->type == OMPSTMT)
			{
				tptr = &((*tptr)->body->body->u.omp);
				continue;
			}
		}
		if ((*tptr)->body->type != OMPSTMT) return 0;
		tptr = &((*tptr)->body->u.omp);
		level++;
	}

	return level;
}

/**
 * Produce N copies of a given OMP tree, where N = # of different 
 * modules, then split/combine and transform the copies. These tasks
 * are performed following the declared transformation rules.
 *
 * @param t The input tree
 */
static
void ast_omp_xform_withrules(aststmt *t)
{
	int savecpl = cur_parallel_line, savectgl = cur_taskgroup_line;
	
	setelem(xformrules) e;
	
	ompdirt_e type = (*t)->u.omp->type;

	if (empty_bodied_omp(t))
		return;

	if (is_target_related(type))
		CHECKTARGET();
	
	/* Transform target constructs multiple times (1 transformation/device) */
	if (is_target_construct(type))
	{
		ast_xfr_gen_tree_copies(t);
		for (e = ast_xfrules->first; e; e = e->next)
		{
			if (!strstr(MODULES_CONFIG, e->key) && strcmp(e->key, DEFAULTDEVICE))
				continue;

			if (!e->value->rules[type].initialized)
				continue; 

			split_combine_xform(e, type, e->value->vars->tree);
		}
		ast_xfr_free_tree_copies();
	}
	else
	{
		/* When in a GPU function transformation, always select
		 * the CUDA transformation. 
		 */
		if (ingpufuncdef)
		{
			e = set_get(ast_xfrules, "cuda");
			if (!e)
				e = ast_xfr_get_defaultrules();
		}
		else
		{
			if (stack_isempty(ompdirectives))
				e = ast_xfr_get_defaultrules();
			else 
				e = set_get(ast_xfrules, XFORM_CURR_DIRECTIVE->modulename);
		}
		

		split_combine_xform(e, type, t);
	}
	
	cur_taskgroup_line = savectgl;
	cur_parallel_line = savecpl;
}

void ast_omp_xform(aststmt *t)
{
	/* First do a check on the clauses */
	xc_validate_clauses((*t)->u.omp->type, (*t)->u.omp->directive->clauses);

	switch ((*t)->u.omp->type)
	{
		case DCTASKGROUP:
			if (!empty_bodied_omp(t))
			{
				int save_tgl = cur_taskgroup_line;
				
				cur_taskgroup_line = (*t)->l;    /* save */
				xform_ompcon_body((*t)->u.omp);
				xform_taskgroup(t);
				cur_taskgroup_line = save_tgl;   /* restore */
			}
			break;
		case DCTARGETDATA:
		{
			int bak;
			
			if (empty_bodied_omp(t))
				break;

			CHECKTARGET();

			/* We store the scope level of the target data and use it later in
			 * xform_ompcon_body -> declare_private_dataclause_vars to set
			 * isindevenv
			 */
			bak = target_data_scope;  /* backup */
			target_data_scope = stab->scopelevel;

			xform_ompcon_body((*t)->u.omp);
			xform_targetdata(t);

			target_data_scope = bak;  /* restore */
			break;
		}
		case DCDECLTARGET:
		{
			if (!(*t)->u.omp->directive->clauses)     /* Old (v4.0) style */
			{ 
				if (empty_bodied_omp(t))
					break;
				
				inDeclTarget++;
				ast_stmt_xform(&((*t)->u.omp->body));   /* First transform the body */
				inDeclTarget--;
			}
			xform_declaretarget(t);
			break;
		}
		case DCFOR:
		case DCFOR_P:
		case DCSINGLE:
		case DCSECTIONS:
			xform_ompcon_body((*t)->u.omp);
			goto XFORM_POINT;
		case DCDISTRIBUTE:
			xform_distribute(t, dist_combined, NULL);
			dist_combined = 0;
			break;
		case DCDISTPARFOR:
			xform_distparfor(t, 1);
			dist_combined = 0;
			break;
		case DCSECTION:
			/* To avoid falling into default case; Handled within
			 * xform_ompcon_body of DCSECTIONS case.
			 */
			ast_ompcon_xform(&((*t)->u.omp));
			break;
		case DCCRITICAL:
		case DCMASTER:
		case DCORDERED:
		case DCATOMIC:
			ast_stmt_xform(&((*t)->u.omp->body));     /* First transform the body */
			goto XFORM_POINT;
		case DCTEAMSDISTPARFOR:
		case DCTARGETTEAMSDISTPARFOR:
			dist_combined = 1;
		case DCTARGETTEAMS:  
		default:
		{
			XFORM_POINT:

			if (ast_xfr_type_issupported((*t)->u.omp->type))
			{
				ast_omp_xform_withrules(t);
				break;
			}
			
			fprintf(stderr, "WARNING: %s directive not implemented\n",
			        ompdirnames[(*t)->u.omp->type]);
			ast_ompcon_xform(&((*t)->u.omp));
			break;
		}
	}
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     TRANSFORM THE AST (ITS OPENMP NODES)                      *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


void ast_xform(aststmt *tree)
{
	if (__has_target && analyzeKernels)
		cars_analyze_declared_funcs(*tree);   /* CARS analysis */
	decltarg_find_all_directives(*tree);    /* Get all #declare target ids */
	
	stack_init(ompdirs, &ompdirectives);
	set_init(fbstmts, &funcbodycp);
	ast_stmt_xform(tree);     /* Core */
	
	/* Add "ort_*_required" functions to constructor */
	add_reqfuncs();
	
	bld_outfuncs_xform();     /* Transform & add the new thread/task funcs */
	bld_outfuncs_place();     /* Place them wisely */
	sgl_fix_sglvars();        /* Adds something to the tail */
	xkn_produce_decl_var_code();  /* Adds to globals and tail */
	bld_ctors_build();        /* Autoinits */
	bld_headtail_place(tree); /* Finish up the tree */
	stack_free(ompdirectives);
	
}
