/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* stack.h - definition and implementation of */
/* a (non-thread safe) stack of a user-defined data type. */

#ifndef __STACK_H__
#define __STACK_H__

#include <assert.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>

/*
 * 1. Usage
 * 
 * 1.1 Stack declaration (and initialization)
 * 
 * To declare a stack of type `TYPENAME', use the following syntax:
 * 
 *   stack(TYPENAME) mystack;
 * 
 * To initialize a stack, you use `stack_new(TYPENAME)':
 * 
 *   mystack = stack_new(TYPENAME);
 * 
 * Declaration and initialization of a global stack, outside of a 
 * function can be also made through the following syntax:
 * 
 *   stack(TYPENAME) mystack = STACK_INITIALIZER;
 * 
 * 1.2 Iteration
 * 
 * To iterate over the elements of the stack, use the following
 * code:
 * 
 *   stackelem(TYPENAME) el;
 *   for (el = mystack->top; el; el = el->prev)
 *   ...
 * 
 * 1.3 Available Functions
 * 
 * The available stack functions are the following:
 * 
 *   stack_new(NAME)
 *   stack_init(NAME,stack)
 *   stack_isempty(stack)
 *   stack_push(stack, element)
 *   stack_pop(stack)
 *   stack_peep(stack)
 *   stack_find(stack, searchfunc)
 *   stack_drain(stack)
 *   stack_free(stack)
 *   stack_size(stack)
 *   stack_print(stack, tostring)
 * 
 */

#define stack(NAME) _##NAME##stack
#define stackelem(NAME) _##NAME##stackelem
#define stacktype(NAME) _##NAME##stacktype
#define SEARCHFUNC_PROTO(NAME) int (*searchfunc)(stacktype(NAME))
#define TOSTRING_PROTO(NAME)   void (*tostring)(stacktype(NAME),char**)

/************************************************************
 *                                                          *
 *                     STACK DEFINITION                     *
 *                                                          *
 ************************************************************/
/**
 * STACK_TYPE_DEFINE generates the structs and the prototypes 
 * for the functions.
 *
 * It should be called from a .h file and it should be accessible 
 * (visible) from all the files that are using the stack.
 *
 * @param NAME     A name given to the stack type. It is used for creating
 *                 new stacks and stack elements.
 * @param DATATYPE The type of the data stored in the stack.
 * 
 */
#define STACK_TYPE_DEFINE(NAME, DATATYPE) \
typedef struct _##NAME##stackelem_ \
{\
	DATATYPE data;\
	struct _##NAME##stackelem_ *prev;\
} *_##NAME##stackelem;\
typedef DATATYPE _##NAME##stacktype;\
typedef struct _##NAME##stack_ \
{\
	int nelems;\
	stackelem(NAME) top;\
	struct _##NAME##stackfuncs *__funcs;\
} *_##NAME##stack;\
struct _##NAME##stackfuncs\
{\
	int             (*isempty)  (stack(NAME) s);\
	void            (*try_init) (stack(NAME) *stack);\
	stackelem(NAME) (*push) (stack(NAME) s, stacktype(NAME) elem);\
	stackelem(NAME) (*pop)  (stack(NAME) s);\
	stackelem(NAME) (*peep) (stack(NAME) s);\
	stackelem(NAME) (*find) (stack(NAME) s, SEARCHFUNC_PROTO(NAME));\
	void            (*drain)    (stack(NAME) s);\
	void            (*copy)     (stack(NAME) s, stack(NAME) *dest);\
	void            (*free)     (stack(NAME) s);\
	void            (*print)    (stack(NAME) s, TOSTRING_PROTO(NAME));\
};\
\
int             (_##NAME##stack_isempty)  (stack(NAME) s);\
void            (_##NAME##stack_try_init) (stack(NAME) *stack);\
stackelem(NAME) (_##NAME##stack_push) (stack(NAME) s, stacktype(NAME) elem);\
stackelem(NAME) (_##NAME##stack_pop)  (stack(NAME) s);\
stackelem(NAME) (_##NAME##stack_peep) (stack(NAME) s);\
stackelem(NAME) (_##NAME##stack_find) (stack(NAME) s, SEARCHFUNC_PROTO(NAME));\
void            (_##NAME##stack_drain)    (stack(NAME) s);\
void            (_##NAME##stack_copy)     (stack(NAME) s, stack(NAME) *dest);\
void            (_##NAME##stack_free)     (stack(NAME) s);\
void            (_##NAME##stack_print)    (stack(NAME) s, TOSTRING_PROTO(NAME));

/************************************************************
 *                                                          *
 *                   STACK IMPLEMENTATION                   *
 *                                                          *
 ************************************************************/
/**
 * STACK_TYPE_IMPLEMENT generates the implementation of each stack function
 *
 * It should be called from a .c file.
 *
 * @param NAME  A name given to the stack type. It is used for creating
 *              new stacks and stack elements.
 * 
 */
#define STACK_TYPE_IMPLEMENT(NAME) \
static struct _##NAME##stackfuncs _##NAME##__funcs = {\
	_##NAME##stack_isempty,\
	_##NAME##stack_try_init,\
	_##NAME##stack_push,\
	_##NAME##stack_pop,\
	_##NAME##stack_peep,\
	_##NAME##stack_find,\
	_##NAME##stack_drain,\
	_##NAME##stack_copy,\
	_##NAME##stack_free,\
	_##NAME##stack_print\
};\
struct _##NAME##stack_ _##NAME##stackinitializer = {\
	0, NULL, &_##NAME##__funcs\
};\
stackelem(NAME) new##_##NAME##stackelem()\
{\
	stackelem(NAME) s = \
		(stackelem(NAME)) malloc(sizeof(struct _##NAME##stackelem_));\
	return s;\
}\
/* Create a new stack */\
stack(NAME) _##NAME##stack_new()\
{\
	stack(NAME) stack = \
		(_##NAME##stack) calloc(1,sizeof(struct _##NAME##stack_));\
	stack->nelems = 0;\
	stack->top = NULL;\
	stack->__funcs = &_##NAME##__funcs;\
	return stack;\
}\
/* Create or drain a stack */\
void _##NAME##stack_init(stack(NAME) *stack)\
{\
	if (*stack == NULL)\
		*stack = stack_new(NAME);\
	else\
		stack_drain(*stack);\
}\
/* Initialize a stack declared with STACK_INITIALIZER */\
void _##NAME##stack_try_init(stack(NAME) *stack)\
{\
	if (*stack == &_##NAME##stackinitializer)\
		*stack = stack_new(NAME);\
}\
/* Check if stack is empty */\
int _##NAME##stack_isempty(stack(NAME) stack)\
{\
	return (stack->nelems == 0) && (stack->top == NULL);\
}\
/* Push an element to a stack */\
stackelem(NAME) _##NAME##stack_push(stack(NAME) s, stacktype(NAME) data)\
{\
	stackelem(NAME) elem;\
	elem = new##_##NAME##stackelem();\
	elem->data = data;\
	if (s->nelems == 0)\
		elem->prev = NULL;\
	else \
		elem->prev = s->top;\
	s->top = elem;\
	++(s->nelems);\
	return elem;\
}\
/* Pop an element from a stack */\
stackelem(NAME) _##NAME##stack_pop(stack(NAME) s)\
{\
	stackelem(NAME) elem;\
	if (_##NAME##stack_isempty(s))\
		return NULL;\
	elem = s->top;\
	s->top = s->top->prev;\
	elem->prev = NULL;\
	--(s->nelems);\
	return elem;\
}\
/* Peep a stack */\
stackelem(NAME) _##NAME##stack_peep(stack(NAME) s)\
{\
	if (_##NAME##stack_isempty(s))\
		return NULL;\
	return s->top;\
}\
/*\
 * Search for an element in the stack.\
 * \
 * @param s           The stack.\
 * @param searchfunc  Prototype: see SEARCHFUNC_PROTO macro.\
 *                    A function that returns 1 if the given parameter\
 *                    (stack element data) satisfies the search criteria.\
 */\
stackelem(NAME) _##NAME##stack_find(stack(NAME) s, SEARCHFUNC_PROTO(NAME))\
{\
	stackelem(NAME) elem;\
	for (elem = s->top; elem; elem = elem->prev)\
		if (searchfunc(elem->data))\
			return elem;\
	return NULL;\
}\
/* Drain a stack */\
void _##NAME##stack_drain(stack(NAME) s)\
{\
	stackelem(NAME) elem;\
	for (;s->top;)\
	{\
		elem = s->top;\
		s->top = elem->prev;\
		elem->prev = NULL;\
		free(elem);\
	}\
	s->nelems = 0;\
	s->top = NULL;\
}\
/* Copy a stack to a destination stack */\
void _##NAME##stack_copy(stack(NAME) s, stack(NAME) *dest)\
{\
	stackelem(NAME) elem;\
	stack(NAME) dummystack = stack_new(NAME);\
	stack_init(NAME, dest);\
	if (_##NAME##stack_isempty(s))\
		return;\
	(*dest)->nelems = 0;\
	(*dest)->top = NULL;\
	(*dest)->__funcs = &_##NAME##__funcs;\
	for (elem = s->top; elem; elem=elem->prev)\
		stack_push(dummystack, elem->data);\
	for (elem = dummystack->top; elem; elem=elem->prev)\
		stack_push(*dest, elem->data);\
	stack_free(dummystack);\
}\
/* Free up memory allocated for a stack */\
void _##NAME##stack_free(stack(NAME) s)\
{\
	stack_drain(s);\
	free(s);\
}\
/*\
 * Dump the contents of a stack\
 * \
 * @param s           The stack.\
 * @param tostring    Prototype: see TOSTRING_PROTO macro.\
 *                    A function that modifies a given string so as\
 *                    to contain the serialized version of a stack element's\
 *                    data.\
 */\
void _##NAME##stack_print(stack(NAME) s, TOSTRING_PROTO(NAME))\
{\
	stackelem(NAME) elem;\
	int i = s->nelems;\
	printf("stack(%s, %d) = {\n", #NAME, i);\
	if (_##NAME##stack_isempty(s))\
	{\
		printf("(empty)\n");\
		printf("}\n");\
		return;\
	}\
	char *buf = malloc(256);\
	for (elem = s->top; elem; elem = elem->prev, --i)\
	{\
		tostring(elem->data, &buf);\
		printf("  elem(%d): %s", i-1, buf);\
		printf("%s\n", i!=1?",":"");\
	}\
	free(buf);\
	printf("}\n");\
}
#define STACK_INITIALIZER(NAME) &_##NAME##stackinitializer
#define stack_new(NAME)        _##NAME##stack_new()
#define stack_init(NAME,stack) _##NAME##stack_init(stack)
#define Newstackelem(NAME)     new##_##NAME##stackelem()

#define STACKTYPE(NAME,DATATYPE)\
	STACK_TYPE_DEFINE(NAME,DATATYPE);\
	STACK_TYPE_IMPLEMENT(NAME);

#define stack_isempty(s)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->isempty(s))
#define stack_push(s,e)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->push(s,e))
#define stack_pop(s)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->pop(s))
#define stack_peep(s)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->peep(s))
#define stack_find(s,sf)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->find(s,sf))
#define stack_drain(s)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->drain(s))
#define stack_copy(s,d)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->copy(s,d))
#define stack_free(s)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->free(s), (s) = NULL)
#define stack_size(s)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->nelems)
#define stack_print(s,t)\
	(assert(s != NULL), (s)->__funcs->try_init(&s), (s)->__funcs->print(s,t))

#endif
