/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* rt_common.h -- runtime facilities common among host and devices */

#ifndef __RT_COMMON_H__
#define __RT_COMMON_H__

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include "omp.h"

typedef unsigned long u_long;
typedef unsigned long long u_long_long;

#ifdef HAVE_LONG_LONG_INT
	#define P2_21M1 (u_long_long) 2097151
	#define _MIN(X,Y) (((X) > (Y)) ? (Y) : (X))


	static inline u_long_long _ull_encode3(unsigned long x,
		                               unsigned long y, unsigned long z)
	{
		return (_MIN(z,P2_21M1) << 42) | (_MIN(y,P2_21M1) << 21) | (_MIN(x,P2_21M1));
	}
	static inline void _ull_decode3(u_long_long n, unsigned long *x, 
		                            unsigned long *y, unsigned long *z)
	{
		*x = _MIN(n & P2_21M1, P2_21M1);
		*y = _MIN((n >> 21) & P2_21M1, P2_21M1);
		*z = _MIN((n >> 42) & P2_21M1, P2_21M1);
	}
#else
	#define _ull_encode3(x,y,z)
	#define _ull_decode3(n,x,y,z)
#endif

/* LOCK TYPES 
 */
#define ORT_LOCK_NORMAL 0
#define ORT_LOCK_NEST   1
#define ORT_LOCK_SPIN   2

typedef struct
{
	omp_sched_t schedule;
	u_long      chunksize;
	u_long      chunk_index;
	int         first_chunk;
	int         mean;
	int         overhead;
	double      sigma;
	double      alpha;
	int        *time_table;
} xsched_data_t;


/* ICVs (internal control variables -- OpenMP V.4.0) 
 */

/* Per-task ICVs */
typedef struct
{
	int         dynamic;
	int         nested;
	omp_sched_t rtschedule;     /* For runtime schedules */
	u_long      rtchunk;        /* ditto */
	int         nthreads;       /* default # threads for a team */
	/* OpenMP 4.0 */
	int         threadlimit;
	omp_proc_bind_t proc_bind;
	int         def_device;
	void       *cur_de;         /* current device data environment */
	
	xsched_data_t xsched;
} ort_task_icvs_t;

/* Global ICVs */
typedef struct
{
	int         dynamic;        /* For the initial task */
	int         nested;
	omp_sched_t rtschedule;     /* For runtime schedules */
	int         rtchunk;        /* ditto */
	int         nthreads;       /* default # threads for a team */

	int         ncpus;          /* Global */
	size_t      stacksize;
	int         waitpolicy;
	int         threadlimit;
	int         levellimit;

	/* OpenMP 4.0 */
	omp_proc_bind_t proc_bind;
	int         cancel;
	int         def_device;
	/* OpenMP 4.5 */
	int         max_task_prio;
	 /* OpenMP 5.0 */
	int         display_affinity;  /* Global */
	char       *affinity_format;   /* per device */
	int         targetoffload;     /* Global */

	xsched_data_t xsched;
} ort_icvs_t;

/* PLACES
 */
#define numplaces(p)   ((p) ? (p)[0][0] : 0)
#define placelen(p,i)  ((p) ? (((i) >= 0) && ((i) < (p)[0][0]) ? \
                           ( (p)[(i)+1] ? (p)[(i)+1][0] : 0 ) : 0) : 0)
int  **places_dup(int **from);
void   places_free(int **place);

/* THREAD/TASK/KERNEL (TTK) Functions
 * (for MPI-enabled OMPi)
 */

/* Handy type */
typedef void *(*ttkfunc_t)(void *);

typedef struct  {
					ttkfunc_t func;     /* function pointer */
        	char      *name;    /* function name */
        } ttkfuncdata_t;

typedef struct {
        	ttkfuncdata_t *table;
        	int           size;            /* used space */
        	int           capacity;        /* total allocated space */
        } ttkfunctable_t;

/* Generic functions */
extern void ttkfunctable_add(ttkfunctable_t *t, ttkfunc_t func, char *name);
extern int  ttkfunctable_findbyptr(ttkfunctable_t *t, ttkfunc_t func);
extern int  ttkfunctable_findbyname(ttkfunctable_t *t, char *name);
extern ttkfunc_t ttkfunctable_getptr(ttkfunctable_t *t, int index);

/* Kernel-specific functions */
extern void ort_kernfunc_register(char *name, ttkfunc_t funcptr);
extern int ort_kernfunc_findbyname(char *name);
extern ttkfunc_t ort_kernfunc_getptr(int id);
extern void ort_kernfunc_cleanup(void);

#endif /* __RT_COMMON_H__ */
